/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.pde.internal.ui.wizards;

import java.util.Iterator;

import org.eclipse.core.runtime.*;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.wizard.*;
import org.eclipse.pde.internal.ui.elements.*;
import org.eclipse.pde.ui.IPluginContentWizard;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;


public abstract class WizardListSelectionPage extends BaseWizardSelectionPage
		implements ISelectionChangedListener, IExecutableExtension {
	protected TableViewer wizardSelectionViewer;
	protected ElementList wizardElements;
	private WizardSelectedAction doubleClickAction = new WizardSelectedAction();
	
	private class WizardSelectedAction extends Action {
		public WizardSelectedAction() {
			super("wizardSelection"); //$NON-NLS-1$
		}
		public void run() {
			selectionChanged(new SelectionChangedEvent(wizardSelectionViewer,
					wizardSelectionViewer.getSelection()));
			advanceToNextPage();
		}
	}
	public WizardListSelectionPage(ElementList wizardElements, String message) {
		super("ListSelection", message); //$NON-NLS-1$
		this.wizardElements = wizardElements;
	}
	public void advanceToNextPage() {
		getContainer().showPage(getNextPage());
	}
	public ElementList getWizardElements() {
		return wizardElements;
	}
	
	public void createControl(Composite parent) {
		Composite container = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.verticalSpacing = 10;
		container.setLayout(layout);
		container.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		createAbove(container, 1);
		Label label = new Label(container, SWT.NONE);
		label.setText(getLabel());
		GridData gd = new GridData();
		label.setLayoutData(gd);
		
		SashForm sashForm = new SashForm(container, SWT.HORIZONTAL);
		sashForm.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		wizardSelectionViewer = new TableViewer(createTable(sashForm, SWT.BORDER));
		wizardSelectionViewer.setContentProvider(new ListContentProvider());
		wizardSelectionViewer.setLabelProvider(ListUtil.TABLE_LABEL_PROVIDER);
		wizardSelectionViewer.setSorter(ListUtil.NAME_SORTER);
		wizardSelectionViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				doubleClickAction.run();
			}
		});
		createDescriptionIn(sashForm);
		createBelow(container, 1);
		initializeViewer();		
		wizardSelectionViewer.setInput(wizardElements);
		wizardSelectionViewer.addSelectionChangedListener(this);		
		Dialog.applyDialogFont(container);
		setControl(container);
	}
	
	protected void createAbove(Composite container, int span) {
	}
	protected void createBelow(Composite container, int span) {
	}
	
	protected void initializeViewer() {
	}
	
	private Table createTable(Composite parent, int style) {
		Table table = new Table(parent, style);
		new TableColumn(table, SWT.NONE);
		TableLayout layout = new TableLayout();
		layout.addColumnData(new ColumnPixelData(200));
		table.setLayout(layout);
		return table;
	}
	
	public void selectionChanged(SelectionChangedEvent event) {
		setErrorMessage(null);
		IStructuredSelection selection = (IStructuredSelection) event.getSelection();
		WizardElement currentWizardSelection = null;
		Iterator iter = selection.iterator();
		if (iter.hasNext())
			currentWizardSelection = (WizardElement) iter.next();
		if (currentWizardSelection == null) {
			setDescriptionText(""); //$NON-NLS-1$
			setSelectedNode(null);
			return;
		}
		final WizardElement finalSelection = currentWizardSelection;
		setSelectedNode(createWizardNode(finalSelection));
		setDescriptionText((String) finalSelection.getDescription());
		getContainer().updateButtons();
	}
	
	public IWizardPage getNextPage(boolean shouldCreate) {
		if (!shouldCreate)
			return super.getNextPage();
		IWizardNode selectedNode = getSelectedNode();
		selectedNode.dispose();
		IWizard wizard = selectedNode.getWizard();
		if (wizard == null) {
			super.setSelectedNode(null);
			return null;
		}
		if (shouldCreate)
			// Allow the wizard to create its pages
			wizard.addPages();
		return wizard.getStartingPage();
	}
	
	protected void focusAndSelectFirst() {
		Table table = wizardSelectionViewer.getTable();
		table.setFocus();
		TableItem[] items = table.getItems();
		if (items.length > 0) {
			TableItem first = items[0];
			Object obj = first.getData();
			wizardSelectionViewer.setSelection(new StructuredSelection(obj));
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.core.runtime.IExecutableExtension#setInitializationData(org.eclipse.core.runtime.IConfigurationElement, java.lang.String, java.lang.Object)
	 */
	public void setInitializationData(IConfigurationElement config, String propertyName, Object data) throws CoreException {
	}
	
	public IPluginContentWizard getSelectedWizard() {
		IWizardNode node = getSelectedNode();
		if (node != null)
			return (IPluginContentWizard)node.getWizard();
		return null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.WizardSelectionPage#canFlipToNextPage()
	 */
	public boolean canFlipToNextPage() {
		IStructuredSelection ssel = (IStructuredSelection)wizardSelectionViewer.getSelection();
		return ssel != null && !ssel.isEmpty();
	}
}
