/* File: Open.m - (Interactive) Unix shell version of OpenPanel
 *
 * By: Christopher Lane (lane@sumex-aim.stanford.edu)
 *
 * Date: 10 November 1992
 *
 * Modified:
 *	[17-Feb-96 TRH] Fix for Application option munching (like -NXHost).
 *	[12-Mar-96 TRH] add -V -H -A options; add EditMenu keyEquiv support.
 *
 * Copyright: 1990, 1991 & 1992 by The Leland Stanford Junior University.
 * This program may be distributed without restriction for non-commercial use.
 */

#define USES_APPICON
#define USES_KEYEQUIV
#import "ShellPanel.h"

#define USAGE "\
usage: %s [-f file] [-d directory] [-e type [-e type]] [-p prompt] [-t title] [-i icon] [-m] [-c] [-V] [-H]\n"

#define HELP "\
options:\n\
 -f file      initial filename to display in the name field\n\
 -d directory initial directory to display in the browser\n\
 -e type      type (extension) of files to accept; multiple allowed\n\
 -p prompt    set prompt (default \"Name:\")\n\
 -t title     set title (default \"%s\")\n\
 -i iconfile  pathname of application icon to use (default none)\n\
 -m           allow multiple selections\n\
 -c           choose directories\n\
 -V           print version string and exit\n\
 -H           this help\n\
"

#define SEPARATOR "/"

#define MAX_TYPES 64


const char *fullPath(const char *parent, const char *name)
{
	static char buffer[MAXPATHLEN];

	if(parent == NULL) return name;

	if(strcmp(strcpy(buffer, parent), SEPARATOR) != 0) (void) strcat(buffer, SEPARATOR);

	return strcat(buffer, name);
}

void main(int argc, char *argv[])
{
	const char *const *filenames;
	const char *prompt = NULL;
	const char *title = NULL;
	const char *iconFile = NULL;
	int option, status = EXIT_SUCCESS, ntypes = 0;
	const char *name = NULL, *path = NULL, *types[MAX_TYPES];
	BOOL chooseDirectories = NO;
	BOOL allowMultipleFiles = NO;

	NXApp = [Application new];
	argc = NXArgc, argv = NXArgv;	// Application may munch options.

	while ((option = getopt(argc, argv, "f:d:e:p:t:i:mcVH")) != EOF)
		switch (option) {
		case 'f': name = optarg; break;
		case 'd': path = optarg; break;
		case 'e': types[ntypes++] = optarg; break;
		case 'p': prompt = optarg; break;
		case 't': title = optarg; break;
		case 'i': iconFile = optarg; break;
		case 'm': allowMultipleFiles = YES; break;
		case 'c': chooseDirectories = YES; break;
		case 'V': status = EXIT_VERSION; break;
		case 'H': status = EXIT_HELP; break;
		default : status = EXIT_USAGE;
		}
	if (optind < argc) status = EXIT_USAGE;

	types[ntypes] = NULL;

	HandleUsageHelpVersion(status, USAGE, HELP);

	SetAppIconToFile(iconFile);
	ProvideKeyEquivalents();

	{
		OpenPanel *panel = [OpenPanel new];
		int context = [NXApp activateSelf:YES];

		if (prompt) [panel setPrompt:prompt];
		if (title) [panel setTitle:title];
		[panel allowMultipleFiles:allowMultipleFiles];
		[panel chooseDirectories:chooseDirectories];

		if ([panel runModalForDirectory:path file:name types:types]) {
			path = [panel directory];
			for (filenames = [panel filenames]; filenames && *filenames; filenames++)
				(void) puts(fullPath(path, *filenames));
		}
		else status = EXIT_FAILURE;

		if (context) (void) [NXApp activate:context];

		[panel free];
	}
	[NXApp free];

	exit(status);
}
