/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- December 1997
 */

/*
 * CvsPrefs.cpp --- class to handle CVS settings
 */

#include "stdafx.h"

#include <string.h>
#include <stdlib.h>
#include "CvsPrefs.h"
#include "Authen.h"
#include "AppConsole.h"

#ifdef macintosh
#	define PCVS "\pCVSROOT"
#	include <Resources.h>
#	include <Folders.h>
#	include <Errors.h>
#	include <TextUtils.h>
#	include "VolsPaths.h"
#	include "MacMisc.h"
#endif /* macintosh */

#ifdef qQT
#	ifdef HAVE_UNISTD_H
#	include <unistd.h>
#	endif
#	include <pwd.h>
#endif /* qQT */

#ifdef WIN32
#	ifdef _DEBUG
#	define new DEBUG_NEW
#	undef THIS_FILE
	static char THIS_FILE[] = __FILE__;
#	endif
#endif /* WIN32 */

CvsPrefs gCvsPrefs;

static CPersistentT<bool> gCheckoutRO("P_CheckoutRO", true);
static CPersistentT<bool> gZ9Option("P_Z9Option", false);
static CPersistentT<bool> gPruneOption("P_PruneOption", true);
static CPersistentT<bool> gQuietOption("P_QuietOption", false);
static CPersistentT<ISO8559> gIsoConvert("P_IsoConvert", ISO8559_none);
static CPersistentT<bool> gICTextOption("P_ICTextOption", false);
static CPersistentT<bool> gAddBinControl("P_AddBinControl", true);
static CPersistentT<MACENCODING> gMacBinEncoding("P_MacBinEncoding", MAC_HQX);
static PCPStr gCvsroot("P_Cvsroot");
static PCPStr gCvsroot1("P_Cvsroot1");
static PCPStr gCvsroot2("P_Cvsroot2");
static PCPStr gCvsroot3("P_Cvsroot3");
static PCPStr gCvsroot4("P_Cvsroot4");
static PCPStr gCvsroot5("P_Cvsroot5");
static CPersistentT<int> gKserverPort("P_KserverPort", -1);
static CPersistentT<int> gPserverPort("P_PserverPort", -1);
static CPersistentT<int> gRhostPort("P_RhostPort", -1);
static PCPStr gServerName("P_ServerName");
static PCStr gRshName("P_RshName");
#ifndef qQT
static PCStr gHome("P_Home");
#endif /* !qQT */
static PCStr gViewer("P_Viewer", "Notepad");
static PCStr gExtdiff("P_Extdiff", "");
static CPersistentT<bool> gDirtySupport("P_DirtySupport", true);
static CPersistentT<bool> gAlwaysUseCvsroot("P_AlwaysUseCvsroot", false);
static CPersistentT<int> gLogoutTimeOut("P_LogoutTimeOut", 0);
static PCStr gProxyHost("P_ProxyHost", "www.mydomain");
static CPersistentT<int> gProxyPort("P_ProxyPort", 8888);
static CPersistentT<bool> gUseProxy("P_UseProxy", false);

bool CvsPrefs::CheckoutRO(void) const {return gCheckoutRO;}
bool CvsPrefs::Z9Option(void) const {return gZ9Option;}
bool CvsPrefs::PruneOption(void) const {return gPruneOption;}
bool CvsPrefs::QuietOption(void) const {return gQuietOption;}
ISO8559 CvsPrefs::IsoConvert(void) const {return gIsoConvert;}
bool CvsPrefs::ICTextOption(void) const {return gICTextOption;}
bool CvsPrefs::AddControl(void) const {return gAddBinControl;}
MACENCODING CvsPrefs::MacBinEncoding(void) const {return gMacBinEncoding;}
int CvsPrefs::KserverPort(void) const {return gKserverPort;}
int CvsPrefs::PserverPort(void) const {return gPserverPort;}
int CvsPrefs::RhostPort(void) const {return gRhostPort;}
const char * CvsPrefs::ServerName(void) const
{
	return gServerName.empty() ? 0L : (const char *)(*(CPStr *)&gServerName);
}
const char * CvsPrefs::RshName(void) const
{
	return gRshName.empty() ? 0L : (const char *)(*(CPStr *)&gRshName);
}
const char * CvsPrefs::Home(void) const
{
#ifndef qQT
	return gHome.empty() ? 0L : (const char *)(*(CStr *)&gHome);
#else /* qQT */
	static CStr home;
	if (!home.empty())
		return home;

	char *env = getenv ("HOME");
	struct passwd *pw;
	if (env)
		home = env;
	else if ((pw = (struct passwd *) getpwuid (getuid ()))
			 && pw->pw_dir)
		home = pw->pw_dir;
	else
		return 0L;

	return home;
#endif /* qQT */
}
const char * CvsPrefs::Viewer(void) const
{
	return gViewer.empty() ? 0L : (const char *)(*(CStr *)&gViewer);
}
const char * CvsPrefs::ExtDiff(void) const
{
	return gExtdiff.empty() ? 0L : (const char *)(*(CStr *)&gExtdiff);
}
bool CvsPrefs::DirtySupport(void) const {return gDirtySupport;}
int CvsPrefs::LogoutTimeOut(void) const {return gLogoutTimeOut;}
bool CvsPrefs::AlwaysUseCvsroot(void) const {return gAlwaysUseCvsroot;}
const char *CvsPrefs::ProxyHost(void) const {return gProxyHost;}
int CvsPrefs::ProxyPort(void) const {return gProxyPort;}
bool CvsPrefs::UseProxy(void) const {return gUseProxy;}

void CvsPrefs::SetCheckoutRO(bool newState) {gCheckoutRO = newState;}
void CvsPrefs::SetZ9Option(bool newState) {gZ9Option = newState;}
void CvsPrefs::SetPruneOption(bool newState) {gPruneOption = newState;}
void CvsPrefs::SetQuietOption(bool newState) {gQuietOption = newState;}
void CvsPrefs::SetIsoConvert(ISO8559 newState) {gIsoConvert = newState;}
void CvsPrefs::SetICTextOption(bool newState) {gICTextOption = newState;}
void CvsPrefs::SetAddControl(bool newState) {gAddBinControl = newState;}
void CvsPrefs::SetMacBinEncoding(MACENCODING newState) {gMacBinEncoding = newState;}
void CvsPrefs::SetKserverPort(int newState) {gKserverPort = newState;}
void CvsPrefs::SetPserverPort(int newState) {gPserverPort = newState;}
void CvsPrefs::SetRhostPort(int newState) {gRhostPort = newState;}
void CvsPrefs::SetServerName(const char *newservername) {*(CPStr *)&gServerName = newservername;}
void CvsPrefs::SetRshName(const char *newrshname) {*(CPStr *)&gRshName = newrshname;}
void CvsPrefs::SetHome(const char *newhome)
{
#ifndef qQT
	*(CStr *)&gHome = newhome;
#endif /* !qQT */
}
void CvsPrefs::SetViewer(const char *newviewer) {*(CStr *)&gViewer = newviewer;}
void CvsPrefs::SetExtDiff(const char *newdiff) {*(CStr *)&gExtdiff = newdiff;}
void CvsPrefs::SetDirtySupport(bool newState) {gDirtySupport = newState;}
void CvsPrefs::SetLogoutTimeOut(int newTimeout) {gLogoutTimeOut = newTimeout;}
void CvsPrefs::SetAlwaysUseCvsroot(bool newState) {gAlwaysUseCvsroot = newState;}
void CvsPrefs::SetUseProxy(bool useit) {gUseProxy = useit;}
void CvsPrefs::SetProxyHost(const char *newhost) {*(CStr *)&gProxyHost = newhost;}
void CvsPrefs::SetProxyPort(int newport) {gProxyPort = newport;}


bool CvsPrefs::empty(void) const
{
	return gCvsroot.empty();
}

int CvsPrefs::length(void) const
{
	return gCvsroot.length();
}

CvsPrefs::operator const char *() const
{
	return (const char *)(*(CPStr *)&gCvsroot);
}

		// as a C string
CvsPrefs::operator const unsigned char *() const
{
	return (const unsigned char *)(*(CPStr *)&gCvsroot);
}

const char *CvsPrefs::operator=(const char *newstr)
{
	return (*(CPStr *)&gCvsroot) = newstr;
}

const unsigned char *CvsPrefs::operator=(const unsigned char *newstr)
{
	return (*(CPStr *)&gCvsroot) = newstr;
}

CPStr & CvsPrefs::get_cvsroot_list(int entry)
{
	switch(entry)
	{
		case 0:
			return gCvsroot1;
			break;
		case 1:
			return gCvsroot2;
			break;
		case 2:
			return gCvsroot3;
			break;
		case 3:
			return gCvsroot4;
			break;
		case 4:
			return gCvsroot5;
			break;
	}
	return gCvsroot1;
}

bool CvsPrefs::has_cvsroot_in_list(const char *cvsroot, int & entry)
{
	for(int i = 0; i < NUM_CVSROOT; i++)
	{
		CPStr & str = get_cvsroot_list(i);
		if(str.empty())
			continue;
		if(strcmp(str, cvsroot) == 0)
		{
			entry = i;
			return true;
		}
	}
	return false;
}

void CvsPrefs::insert_cvsroot_in_list(const char *cvsroot)
{
	int entry;
	
	if(has_cvsroot_in_list(cvsroot, entry))
	{
		// move to the first entry
		CPStr tmp;
		tmp = get_cvsroot_list(0);
		get_cvsroot_list(0) = get_cvsroot_list(entry);
		get_cvsroot_list(entry) = tmp;
	}
	else
	{
		// push back the entry
		for(int i = (NUM_CVSROOT - 1); i >= 1; i--)
		{
			get_cvsroot_list(i) = get_cvsroot_list(i - 1);
		}
		get_cvsroot_list(0) = cvsroot;
	}
}

void CvsPrefs::flush_list(void)
{
	for(int i = 0; i < NUM_CVSROOT; i++)
	{
		get_cvsroot_list(i) = "";
	}
}

void CvsPrefs::save(void)
{
	if(empty())
		return;
	
	// update the CVSROOT list
	insert_cvsroot_in_list((const char *)*this);
	
	CPersistent::SaveAll();
}

void CvsPrefs::load(void)
{
	// flush the CVSROOT list, it will be filled later
	flush_list();
		
	if(CPersistent::LoadAll())
		return;

	// backward compatibility
#ifdef WIN32
	UINT value;

	CString cstr = AfxGetApp()->GetProfileString(PROFILE_NAME, "CVSROOT");
	if(!cstr.IsEmpty())
		*this = (const char *)cstr;

	if((value = AfxGetApp()->GetProfileInt(PROFILE_NAME, "CheckoutRO", -1)) != -1)
		SetCheckoutRO(value ? 1 : 0);
	if((value = AfxGetApp()->GetProfileInt(PROFILE_NAME, "Z9Option", -1)) != -1)
		SetZ9Option(value ? 1 : 0);
	if((value = AfxGetApp()->GetProfileInt(PROFILE_NAME, "QuietOption", -1)) != -1)
		SetQuietOption(value ? 1 : 0);
	if((value = AfxGetApp()->GetProfileInt(PROFILE_NAME, "PruneOption", -1)) != -1)
		SetPruneOption(value ? 1 : 0);
	if((value = AfxGetApp()->GetProfileInt(PROFILE_NAME, "Authentication", -1)) != -1)
		gAuthen.setkind((AuthenKind)value);
	if((value = AfxGetApp()->GetProfileInt(PROFILE_NAME, "AddControl", -1)) != -1)
		SetAddControl(value ? 1 : 0);

	char tmp[20];
	for(int i = 0; i < NUM_CVSROOT; i++)
	{
		sprintf(tmp, "CVSROOTLIST%d", i);
		cstr = AfxGetApp()->GetProfileString(PROFILE_NAME, tmp);
		if(!cstr.IsEmpty())
			get_cvsroot_list(i) = (const char *)cstr;
	}
#endif /* WIN32 */

#ifdef macintosh
	FSSpec theFolder;
	short FRef;
	Handle hdl;
	OSErr err;
	Str255 thePath;
	
	err = MacGetPrefsFolder(theFolder, thePath);
	if(err != noErr)
		return;

	err = HSetVol(NULL, theFolder.vRefNum, theFolder.parID);
	FRef = OpenResFile(PCVS);
	if(FRef < 0)
		return;

	hdl = GetResource('CVS ', 128);
	if(hdl != NULL)
	{
		Str255 str;
		HLock(hdl);
		BlockMove(*hdl, str, (*hdl)[0] + 1);
		str[str[0] + 1] = '\0';
		*this = (const char *)str + 1;
		HUnlock(hdl);
	}
	hdl = GetResource('CHRO', 128);
	if(hdl != NULL)
	{
		SetCheckoutRO(**hdl);
	}
	hdl = GetResource('Z9OP', 128);
	if(hdl != NULL)
	{
		SetZ9Option(**hdl);
	}
	hdl = GetResource('QUIE', 128);
	if(hdl != NULL)
	{
		SetQuietOption(**hdl);
	}
	hdl = GetResource('POPT', 128);
	if(hdl != NULL)
	{
		SetPruneOption(**hdl);
	}
	hdl = GetResource('AUTH', 128);
	if(hdl != NULL)
	{
		gAuthen.setkind((AuthenKind)**hdl);
	}
	hdl = GetResource('ISOC', 128);
	if(hdl != NULL)
	{
		SetIsoConvert((ISO8559)**hdl);
	}
	hdl = GetResource('ICTO', 128);
	if(hdl != NULL)
	{
		SetICTextOption(**hdl);
	}
	hdl = GetResource('ACTL', 128);
	if(hdl != NULL)
	{
		SetAddControl(**hdl);
	}

	for(int i = 0; i < NUM_CVSROOT; i++)
	{
		hdl = GetResource('CROL', 128 + i);
		if(hdl != NULL)
		{
			HLock(hdl);
			get_cvsroot_list(i) = (const char *)*hdl;
			HUnlock(hdl);
		}
	}
	CloseResFile (FRef);
#endif /* macintosh */
}

#ifdef WIN32
kWinVersion CvsPrefs::WhichVersion()
{
	DWORD dwVersion = GetVersion(); // Get major and minor version numbers of Windows
	DWORD dwWindowsMajorVersion =  (DWORD)(LOBYTE(LOWORD(dwVersion)));
	DWORD dwWindowsMinorVersion =  (DWORD)(HIBYTE(LOWORD(dwVersion)));
	DWORD dwBuild;

	// Get build numbers for Windows NT or Win32s
	if (dwVersion < 0x80000000)                // Windows NT
	{
		dwBuild = (DWORD)(HIWORD(dwVersion));
		return dwWindowsMajorVersion == 3 ? kWinNT35 : kWinNT4;
	}
	else if (dwWindowsMajorVersion < 4)        // Win16s
	{
		dwBuild = (DWORD)(HIWORD(dwVersion));
		return kWin16;
	}
	
	// Windows 95 -- No build numbers provided    dwBuild =  0;
	return kWin32;
}
#endif /* WIN32 */
