/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- December 1997
 */

#include "config.h"

#include <GUSI.h>
#include <TFileSpec.h>
#include <sys/errno.h>
#include "hqx.h"
#include "cvs_hqx.h"
#include "FileCopy.h"
#include "apsingle.h"

#include <PLStringFuncs.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <Resources.h>
#include <TextUtils.h>
#include <Strings.h>
#include <Finder.h>

#include <ICAPI.h>

#if 0
// if you want some files to be handled as plain binary
// (even if the file has a resource fork), open the
// application with ResEdit or Resorcerer and change
// the 'bin#' resource :

static const	OSType		kTypesListResType	= 'bin#';
static const	short		kTypesListResID		= 128;
#endif

extern "C" char *macos_fixpath (const char *path);

static void fillSemiColonTypes(const char *envVar, OSType *& allTypes, int & numTypes)
{
	allTypes = 0L;
	numTypes = 0;
	const char *envRes = getenv(envVar);
	if(envRes == 0L)
		return;
	
	char *tmpStr = (char *)malloc(strlen(envRes) + 1);
	if(tmpStr == 0L)
	{
		fprintf(stderr, "Running out of memory !\n");
		exit(1);
	}
	strcpy(tmpStr, envRes);
	
	char *tmp = tmpStr, *colon = 0L, *start = tmpStr;
	while((tmp = strchr(tmp, ';')) != 0L)
	{
		*tmp = '\0';
		if((tmp - start) == 4)
		{
			if(allTypes == 0L)
				allTypes = (OSType *)malloc(sizeof(OSType));
			else
				allTypes = (OSType *)realloc(allTypes, (numTypes + 1) * sizeof(OSType));
			if(allTypes == 0L)
			{
				fprintf(stderr, "Running out of memory !\n");
				exit(1);
			}
			OSType atype;
			((char *)&atype)[0] = *start++;
			((char *)&atype)[1] = *start++;
			((char *)&atype)[2] = *start++;
			((char *)&atype)[3] = *start;
			allTypes[numTypes++] = atype;
		}
		else
		{
			fprintf(stderr, "Warning unknown signature %s (variable : %s) !\n", start, envRes);
		}
		
		start = ++tmp;
	}
}

static Boolean isPlainBinary(OSType ftype)
{
#if 1
	static OSType *allTypes = (OSType *)-1;
	static int numTypes;
	if(allTypes == (OSType *)-1)
	{
		fillSemiColonTypes("MAC_BINARY_TYPES_PLAIN", allTypes, numTypes);
	}
	if(allTypes == 0L)
		return false;
	for(int i = 0; i < numTypes; i++)
	{
		if(allTypes[i] == ftype)
			return true;
	}
	return false;
#else
	Handle	typesHandle = Get1Resource (kTypesListResType, kTypesListResID);
	if (typesHandle == nil)
	{
		fprintf(stderr, "Unable to get the ""bin#"" resource (error %d) !\n", ResError());
		return false;
	}
	
	HLock(typesHandle);
	Ptr typesPtr = *typesHandle;
	
	short		typesCount = *(short*)typesPtr;
	typesPtr += (2 * sizeof(short)) / sizeof(char); // count + align
	OSType*		walker = (OSType *)typesPtr;

	while (typesCount-- > 0) {
		if (*walker == ftype)
		{
			HUnlock(typesHandle);
			return true;
		}
		walker++;
	}
	HUnlock(typesHandle);
	
	return false;
#endif
}

static Boolean testHQXEncoding(OSType ftype)
{
	static OSType *allTypes = (OSType *)-1;
	static int numTypes;
	if(allTypes == (OSType *)-1)
	{
		fillSemiColonTypes("MAC_BINARY_TYPES_HQX", allTypes, numTypes);
	}
	if(allTypes == 0L)
		return false;
	for(int i = 0; i < numTypes; i++)
	{
		if(allTypes[i] == ftype)
			return true;
	}
	return false;
}

static Boolean testAppleSingleEncoding(OSType ftype)
{
	static OSType *allTypes = (OSType *)-1;
	static int numTypes;
	if(allTypes == (OSType *)-1)
	{
		fillSemiColonTypes("MAC_BINARY_TYPES_SINGLE", allTypes, numTypes);
	}
	if(allTypes == 0L)
		return false;
	for(int i = 0; i < numTypes; i++)
	{
		if(allTypes[i] == ftype)
			return true;
	}
	return false;
}

// - Tells which encoder to use. Check the env. variables "MAC_BINARY_TYPES_HQX"
// and "MAC_BINARY_TYPES_SINGLE". If the type is not inside, return the
// default encoder "MAC_DEFAULT_RESOURCE_ENCODING".
// - Note that the plain binary test has to overide this one in every cases.

typedef enum
{
	eEncodeHQX,
	eEncodeAppleSingle
} encodingPolicy;

static void determineEncodingPolicy(OSType ftype, encodingPolicy *policy)
{
	if(testHQXEncoding(ftype))
	{
		*policy = eEncodeHQX;
		return;
	}
	else if(testAppleSingleEncoding(ftype))
	{
		*policy = eEncodeAppleSingle;
		return;
	}
	
	// get the default encoder
	char *envRes = getenv("MAC_DEFAULT_RESOURCE_ENCODING");
	if(envRes == 0L)
	{
		// default was eEncodeHQX, so MWCVS users won't complain
		// if they don't set this flag
		*policy = eEncodeHQX;
		return;
	}
	
	if(strcmp(envRes, "HQX") == 0)
	{
		*policy = eEncodeHQX;
		return;
	}
	else if(strcmp(envRes, "AppleSingle") == 0)
	{
		*policy = eEncodeAppleSingle;
		return;
	}
	
	fprintf(stderr, "Unknown resource encoding '%s' !\n", envRes);
	exit(1);
}

static FILE *gDestFile;
static FILE *gSrcFile;
static FSSpec gDestFileSpec;

static OSErr MyDest(void* buffer, long count, long param) {
	fwrite(buffer, count, 1, gDestFile);
	return noErr;
}

static OSErr MySource(void* buffer, long *count, long param) {
	*count =  fread(buffer, 1, *count, gSrcFile);
	return noErr;
}

static OSErr MyNameFilter(StringPtr name, short *vol, long *dir, long param) {
	PLstrcpy(name, gDestFileSpec.name);
	*vol = gDestFileSpec.vRefNum;
	*dir = gDestFileSpec.parID;
	return noErr;
}

int
convert_hqx_file (const char *infile,  int inflags,
	const char *outfile, int outflags, int *bin)
{
	OSErr err = fnfErr;
	FInfo fileinfo;
	encodingPolicy inFileType;
	
	TFileSpec inspec(infile);
	if(TFileSpec::Error() != noErr)
		goto panic;

	TFileSpec outspec(outfile);
	if(TFileSpec::Error() != noErr)
		goto panic;

	Boolean encode;
	
	if (inflags & O_BINARY)
	{
		encode = false;
		
		// the cvswrapper has to be coherent !!!
		if((*bin) == 0)
			return 0;
		
		gSrcFile = fopen(infile, "r");
		if(gSrcFile == NULL)
			goto panic;
		
		char buf[100];
		static char hqxheader[] = "(This file must be converted with BinHex 4.0)";
		UInt32 asheader = AS_MAGIC_NUM;
		fread (buf, sizeof(char), 100, gSrcFile);

		// Determine whether the file is one of the known encodings
		if ( memcmp(hqxheader, buf, strlen(hqxheader) * sizeof(char)) == 0)
		{
			inFileType = eEncodeHQX;
			fseek(gSrcFile, 0, SEEK_SET);
		}
		else if  ( memcmp(&asheader, buf, sizeof(asheader)) == 0 )
		{
			fclose(gSrcFile);
			inFileType = eEncodeAppleSingle;
		}
		else
		{
			// it is a "plain" binary
			fclose(gSrcFile);
			return 0;
		}
	}
	else
	{
		encode = true;

		// check if it is a text file
		CInfoPBRec cbrec;
		DirInfo * cb;
		
		cb = (DirInfo *)&cbrec;
		cb->ioNamePtr	=	inspec.name;
		cb->ioDrDirID	=	inspec.parID;
		cb->ioVRefNum	=	inspec.vRefNum;
		cb->ioFDirIndex	=	0;

		err = PBGetCatInfoSync((CInfoPBRec *)cb);
		if(err != noErr)
			goto panic;
		
		fileinfo = ((HFileInfo *)cb)->ioFlFndrInfo;
		if(fileinfo.fdType == 'TEXT')
			return 0;
		
		// Never encode any files that have no resource fork
		// Should we just use the test: if (
		short fileRsrc = FSpOpenResFile((FSSpec *)&inspec, fsRdPerm);
		if(fileRsrc < 0)
		{
			// it is a "plain" binary
			*bin = 1;
			return 0;
		}
		CloseResFile(fileRsrc);
		
		// special case, some files may be forced as plain binary.
		// (env. variable "MAC_BINARY_TYPES_PLAIN")
		if(isPlainBinary(fileinfo.fdType))
		{
			*bin = 1;
			return 0;
		}
	}
	
	if(encode)
	{
		encodingPolicy policy;
		determineEncodingPolicy(fileinfo.fdType, &policy);
		
		if(policy == eEncodeHQX)
		{
			gDestFile = fopen(outfile, "w");
			if(gDestFile == NULL)
				goto panic;
			err = HQXEncode(inspec.name, inspec.vRefNum, inspec.parID, MyDest, 0);
			fflush(gDestFile);
			fclose(gDestFile);
			if(err != noErr)
				goto panic;
		}
		else if ( policy == eEncodeAppleSingle )
		{
#define APPLESINGLE_WANTED_ENTRIES ((UInt32)AS_DATA_BIT | AS_RESOURCE_BIT | AS_COMMENT_BIT | AS_FINDERINFO_BIT | AS_MACINFO_BIT)
			UInt32 wantedEntries = APPLESINGLE_WANTED_ENTRIES;
			err = encodeAppleSingle(&inspec, outfile, wantedEntries);
			if (err != noErr)
				goto panic;
		}
		else
		{
			fprintf( stderr, "Internal error: %s, %s\n", __FILE__, __LINE__);
			goto panic;
		}
	}
	else	// decoding
	{
		if ( inFileType == eEncodeHQX )
		{
			PLstrcpy(gDestFileSpec.name, outspec.name);
			gDestFileSpec.vRefNum = outspec.vRefNum;
			gDestFileSpec.parID = outspec.parID;
			
			err = HQXDecode(MySource, MyNameFilter, true, true, 0);
			fclose(gSrcFile);
			if(err != noErr)
				goto panic;
		}
		else if ( inFileType == eEncodeAppleSingle)
		{
			UInt32 wantedEntries = APPLESINGLE_WANTED_ENTRIES;
			err = decodeAppleSingle(infile, &outspec, wantedEntries);
			if (err != noErr)
				goto panic;

			// Sometimes, type/creator information is not encoded inside the AS file 
			// If so, we want to set type/creator using Internet Config
			CInfoPBRec cbrec;
			cbrec.hFileInfo.ioNamePtr = outspec.name;
			cbrec.hFileInfo.ioDirID	= outspec.parID;
			cbrec.hFileInfo.ioVRefNum =	outspec.vRefNum;
			cbrec.hFileInfo.ioFDirIndex	=0;

			err = PBGetCatInfoSync(&cbrec);
			if(err != noErr)
				goto panic;
			
			fileinfo = cbrec.hFileInfo.ioFlFndrInfo;
			if( (fileinfo.fdType == AS_DEFAULT_TYPE)
				 && ( fileinfo.fdCreator == AS_DEFAULT_CREATOR))
			{
				TFileSpec newSpec( outspec);
				char * fullPath = newSpec.FullPath();
				set_file_type(	fullPath, true );
			}
 		}
		else
		{
			fprintf( stderr, "Internal error: %s, %s\n", __FILE__, __LINE__);
			goto panic;			
		}
	}

	return 1;
panic:
	fprintf(stderr, "Panic error code %d, exiting\n", err);
	exit(1);
	
	return 0;
}

void set_file_type(const char *outfile, Boolean warnOnFail)
{
	OSStatus err;
	ICInstance inst;
	Str255 filename;
	
	strcpy((char *)filename, outfile);
	c2pstr((char *)filename);
	
	err = ICStart(&inst, 'CVS ');			// Use your creator code if you have one!
	if (err != noErr)
	{
		static Boolean firsttime = true;
		if(firsttime)
		{
			fprintf(stderr, "WARNING Internet config's missing !\n");
			firsttime = false;
		}
		return;
	}

	err = ICFindConfigFile(inst, 0, nil);
	if (err != noErr)
	{
		static Boolean firsttime2 = true;
		if(firsttime2)
		{
			fprintf(stderr, "WARNING Internet config file's missing !\n");
			firsttime2 = false;
		}
		goto exitall;
	}
	
	ICMapEntry entry;
	err = ICMapFilename(inst, filename, &entry);
	if (err != noErr)
	{
		if (warnOnFail)
			fprintf(stderr, "WARNING: Internet Config cannot map file %s\n", outfile);
		goto exitall;
	}
	OSErr oserr;
	TFileSpec outspec(outfile);
	FInfo fndrInfo;

	oserr = HGetFInfo(outspec.vRefNum, outspec.parID, outspec.name, &fndrInfo);
	if (oserr != noErr)
		goto exitall;

	if(entry.file_creator != 0)
		fndrInfo.fdCreator = entry.file_creator;
	if(entry.file_type != 0)
		fndrInfo.fdType = entry.file_type;
	oserr = HSetFInfo(outspec.vRefNum, outspec.parID, outspec.name, &fndrInfo);
	if (oserr != noErr)
		goto exitall;

exitall:
	ICStop(inst);
}

int mac_duplicate (const char *from, const char *to)
{
	TFileSpec fromspec(macos_fixpath(from));
	if(fromspec.Error() != noErr)
		return fromspec.Error();
	
	TFileSpec tospec(macos_fixpath(to));
	if(tospec.Error() != noErr && tospec.Error() != fnfErr)
		return tospec.Error();
	
	OSErr err = FileCopy(fromspec.vRefNum,
						 fromspec.parID,
						 fromspec.name,
						 tospec.vRefNum,
						 tospec.parID,
						 nil,
						 tospec.name,
						 nil,
						 0,
						true);
	
	if(err == fnfErr)
		errno = ENOENT;
	else if(err != noErr)
		errno = err;

	return err;
}

unsigned char convert_iso(int convert_from, int iso, unsigned char c)
{
	static Handle iso_to_mac = (Handle)-1;
	static Handle mac_to_iso = (Handle)-1;
	
	if(iso_to_mac == 0L || iso != 1) // only Latin 1 avalaible right now
		return c;
	
	if(iso_to_mac == (Handle)-1)
	{
		iso_to_mac = GetResource('taBL', 1001);
		mac_to_iso = GetResource('taBL', 1002);
		
		if(iso_to_mac == 0L || mac_to_iso == 0L)
		{
			iso_to_mac = 0L;
			mac_to_iso = 0L;
			fprintf(stderr, "WARNING ISO8559 tables are missing !\n");
			return c;
		}
		HNoPurge(iso_to_mac);
		HNoPurge(mac_to_iso);
		HLock(iso_to_mac);
		HLock(mac_to_iso);
	}
	
	return convert_from ? (*iso_to_mac)[c] : (*mac_to_iso)[c];
}
