/*
 * macos_filesys.c
 * Filesystem handling stuff for macos
 *
 * Michael Ladwig <mike@twinpeaks.prc.com> --- November 1995 (Initial version)
 *														 --- May 1996 (Handled relative paths better)
 *														 --- June 1996 (Have open look absolute and relative)
 * Modified : Alexandre Parenteau <aubonbeurre@geocities.com> --- December 1997
 */

#include "cvs.h"
#include <sys/errno.h>

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cvs_hqx.h"

char *macos_fixpath (const char *path);
char scratchPath[1024];

int
macos_mkdir( const char *path, int oflag )
{
	return mkdir( macos_fixpath(path) );
}

int
macos_open( const char *path, int oflag, ... )
{
	int	r;
	char	*macPath;
	
	macPath = macos_fixpath(path);

	r = open( macPath, oflag );
	if( r < 0 ) return open( macPath+1, oflag );

	return r;
}

int
macos_chmod( const char *path, mode_t mode )
{
	return chmod(macos_fixpath(path), mode);
}

int
macos_creat( const char *path, mode_t mode )
{
	return creat( macos_fixpath(path) );
}

FILE *
macos_fopen( const char *path, const char *mode )
{
	FILE	*fp;
		
	fp = fopen(macos_fixpath(path), mode);
	
	/* I'm getting ENOTDIR, CVS expects ENOENT */
	
	if( (fp == NULL) && (errno == ENOTDIR) ) errno = ENOENT;
	
	return fp;
}

int
macos_chdir( const char *path )
{
	int	r;
	char	*macPath;
	
	macPath = macos_fixpath(path);

	r = chdir(macPath+1);
	if( r < 0 ) return chdir(macPath);

	return r;
}

int
macos_access(const char *path, int amode)
{	
	return access( macos_fixpath(path), amode );
}

DIR *
macos_opendir(const char *path)
{
	return opendir( macos_fixpath(path) );
}

int
macos_stat (const char *path, struct stat *ststr)
{
	return stat( macos_fixpath(path), ststr );
}

int
macos_rename (const char *path, const char *newpath)
{
	char	macPath_from[1024], macPath_to[1024];

	strcpy( macPath_from, macos_fixpath(path) );
	strcpy( macPath_to, macos_fixpath(newpath) );

	return rename( macPath_from, macPath_to );
}

int
macos_rmdir (const char *path)
{
	return rmdir( macos_fixpath(path) );
}

int
macos_unlink (const char *path)
{
	chmod(path, S_IRWXU);
	return unlink( macos_fixpath(path) );
}

char *
macos_fixpath (const char *path)
{
	char		*sepCh;
	
	// tip to not make unix->mac path conversion when
	// this should not happen.
	// - this should be obsolete now (4/7/98). Instead
	// gives a Mac path for the HOME env. variable
	// like "System:Preferences" and cvs will convert it
	// back to a Unix path.
	if(strncmp(path, "@@@", 3) == 0)
	{
		strcpy( scratchPath, path + 3 );
		return scratchPath;
	}
	
	// if it is an absolute Unix path, do
	// not prefix by ':'
	if(path[0] != '/')
		strcpy( scratchPath, ":" );
	else
	{
		scratchPath[0] = '\0';
		++path;
	}

	// Unix "./xxx" is equivalent to ":xxx"
	if( (*path == '.') && (*(path+1) == '/') )
		strcat( scratchPath, path+2 );
	// ???
	else if( strcmp(path, ".") != 0 )
		strcat( scratchPath, path );

	// Finally replace '/' by ':'
	while( (sepCh = strchr(scratchPath, '/')) != NULL )
		*sepCh = ':';
		
	//fprintf(stderr,"MacOS fixpath <%s>", path);
	//fprintf(stderr," -> <%s>\n", scratchPath);
		
	return scratchPath;
}

/*
 * the trick is : if inflags & OPEN_BINARY, it means we decode from
 * the server format (HQX or ISO) into the local format (Mac resource or
 * Mac Text encoding). outflags depends if we try to open a Mac CR file
 * or a Unix LF file. The MSL fopen, fread, fwrite... toggles
 * automatically CR<->LF when opening as TEXT ("r", "w"). While
 * the GUSI open, read, write makes no conversion at all (alexp)
 */
	
void
convert_file (char *infile,  int inflags,
	      char *outfile, int outflags, int binary)
{
    FILE *infd, *outfd;
    char buf[8192];
    int len;
    int decode; /* 1 : server to client format, 0: client to server format */
#ifdef BINHEX_STUFF
#	define ISO8559_VALID(iso) ((iso) == 1)
	static int set_text_signature = -1;
	static int convert_iso8559 = -1;
	
	if(convert_iso8559 == -1)
	{
		const char *iso = getenv("ISO8859");
		if(iso == NULL || sscanf(iso, "%d", &convert_iso8559) != 1 ||
			!ISO8559_VALID(convert_iso8559))
				convert_iso8559 = 0;
	}
	if(set_text_signature == -1)
	{
		set_text_signature = getenv("IC_ON_TXT") != NULL ? 1 : 0;
	}
	
	/* "update" doesn't check if we can erase the dest file */
	chmod(outfile, S_IRWXU);
	
	if(convert_hqx_file(infile, inflags, outfile, outflags, &binary))
		return;
#endif /* BINHEX_STUFF */

    if (inflags & OPEN_BINARY) {
    	decode = true;
    } else {
    	decode = false;
    }

    if ((infd = fopen (infile, (binary || (inflags & OPEN_BINARY)) ? "rb" : "r")) == NULL)
        error (1, errno, "couldn't read %s", infile);
    if ((outfd = fopen (outfile, (binary || (outflags & OPEN_BINARY)) ? "wb" : "w")) == NULL)
        error (1, errno, "couldn't write %s", outfile);

    while ((len = fread (buf, sizeof (char), sizeof (buf) / sizeof(char), infd)) > 0) {
		char * conv 	= buf;
		int    convlen  = len;
#ifdef BINHEX_STUFF
		if(binary == 0 && convert_iso8559) while (convlen--)
		{
			*conv = (char)convert_iso(decode, convert_iso8559, (unsigned char)*conv);
			conv++;
		}
#endif /* BINHEX_STUFF */

		if (fwrite (buf, sizeof(char), len, outfd) < 0)
			error (1, errno, "error writing %s", outfile);
    }
    if (len < 0)
        error (1, errno, "error reading %s", infile);

    if (fclose (outfd) != 0)
        error (0, errno, "warning: couldn't close %s", outfile);
    if (fclose (infd) != 0)
        error (0, errno, "warning: couldn't close %s", infile);
        
#ifdef BINHEX_STUFF
	if((binary || set_text_signature) && decode)
		set_file_type(outfile, !quiet);
#endif /* BINHEX_STUFF */
}

