/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- April 1998
 */

/*
 * BrowseFileView.cpp -- Implements the view with all the files
 */

#include "stdafx.h"
#include "wincvs.h"

#include "BrowseFileView.h"
#include "FileTraversal.h"
#include "CvsCommands.h"
#include "MultiFiles.h"
#include "CvsArgs.h"
#include "AppConsole.h"
#include "CvsPrefs.h"
#include "CvsIgnore.h"
#include "TclGlue.h"
#include "MacrosSetup.h"
#include "MoveToTrash.h"
#include "WinCvsBrowser.h"

#include <process.h>
#include <vector>
#include <IO.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// persistent values
CPersistentT<int> gFileViewSort("P_FileViewSort", EntnodeData::kName);
CPersistentT<bool> gFileViewSortAsc("P_FileViewSortAsc", true);
CPersistentT<bool> gFileViewIgnore("P_FileViewIgnore", true);

#define NUM_COLUMNS	7

static _TCHAR *_gszColumnLabel[NUM_COLUMNS] =
{
	_T("Name"), _T("Rev."), _T("Option"), _T("Status"),
	_T("Tag"), _T("Date"), _T("Conflict")
};

static int _gnColumnFmt[NUM_COLUMNS] = 
{
	LVCFMT_LEFT, LVCFMT_LEFT, LVCFMT_LEFT, LVCFMT_LEFT,
		LVCFMT_LEFT, LVCFMT_LEFT, LVCFMT_LEFT
};

static int _gnColumnWidth[NUM_COLUMNS] = 
{
	150, 50, 50, 100, 150, 150, 150
};

/////////////////////////////////////////////////////////////////////////////
// CBrowseFileView

IMPLEMENT_DYNCREATE(CBrowseFileView, CListView)

BEGIN_MESSAGE_MAP(CBrowseFileView, CListView)
	//{{AFX_MSG_MAP(CBrowseFileView)
	ON_COMMAND(ID_VIEW_SMALLICONS, OnViewSmallIcons)
	ON_COMMAND(ID_VIEW_FULLLIST, OnViewList)
	ON_COMMAND(ID_VIEW_ROWDETAILS, OnViewFullRowDetails)
	ON_UPDATE_COMMAND_UI(ID_VIEW_SMALLICONS, OnUpdateViewSmallIcons)
	ON_UPDATE_COMMAND_UI(ID_VIEW_ADD, OnUpdateViewAdd)
	ON_UPDATE_COMMAND_UI(ID_VIEW_ADDB, OnUpdateViewAddB)
	ON_UPDATE_COMMAND_UI(ID_VIEW_COMMIT, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_RELEASE, OnUpdateViewRelease)
	ON_UPDATE_COMMAND_UI(ID_VIEW_RMV, OnUpdateViewRmv)
	ON_UPDATE_COMMAND_UI(ID_VIEW_FULLLIST, OnUpdateViewList)
	ON_UPDATE_COMMAND_UI(ID_VIEW_ROWDETAILS, OnUpdateViewFullRowDetails)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONDBLCLK()
	ON_NOTIFY_REFLECT(LVN_COLUMNCLICK, OnColumnclick)
	ON_COMMAND(ID_VIEW_ADD, OnViewAdd)
	ON_COMMAND(ID_VIEW_ADDB, OnViewAddb)
	ON_COMMAND(ID_VIEW_COMMIT, OnViewCommit)
	ON_COMMAND(ID_VIEW_RMV, OnViewRmv)
	ON_COMMAND(ID_VIEW_UPDATE, OnViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_RELOAD, OnUpdateViewReload)
	ON_COMMAND(ID_VIEW_RELOAD, OnViewReload)
	ON_NOTIFY_REFLECT(LVN_KEYDOWN, OnKeydown)
	ON_COMMAND(ID_VIEW_UPONE, OnViewUpone)
	ON_UPDATE_COMMAND_UI(ID_VIEW_UPONE, OnUpdateViewUpone)
	ON_COMMAND(ID_VIEW_TRASH, OnViewTrash)
	ON_UPDATE_COMMAND_UI(ID_VIEW_TRASH, OnUpdateViewTrash)
	ON_COMMAND(ID_VIEW_DIFF, OnViewDiff)
	ON_COMMAND(ID_VIEW_LOG, OnViewLog)
	ON_COMMAND(ID_VIEW_GRAPH, OnViewGraph)
	ON_COMMAND(ID_VIEW_STATUS, OnViewStatus)
	ON_COMMAND(ID_VIEW_LOCKF, OnViewLock)
	ON_COMMAND(ID_VIEW_UNLOCKF, OnViewUnlock)
	ON_COMMAND(ID_VIEW_WATCHON, OnViewWatchOn)
	ON_COMMAND(ID_VIEW_WATCHOFF, OnViewWatchOff)
	ON_COMMAND(ID_VIEW_EDIT, OnViewEdit)
	ON_COMMAND(ID_VIEW_UNEDIT, OnViewUnedit)
	ON_COMMAND(ID_VIEW_WATCHERS, OnViewWatchers)
	ON_COMMAND(ID_VIEW_EDITORS, OnViewEditors)
	ON_COMMAND(ID_VIEW_RELEASE, OnViewRelease)
	ON_COMMAND(ID_VIEW_TAGNEW, OnViewTagNew)
	ON_COMMAND(ID_VIEW_TAGDELETE, OnViewTagDelete)
	ON_COMMAND(ID_VIEW_TAGBRANCH, OnViewTagBranch)
	ON_WM_RBUTTONDOWN()
	ON_UPDATE_COMMAND_UI(ID_MACRO_SEL, OnUpdateMacroSel)
	ON_COMMAND(ID_VIEW_EXPLORE, OnViewExplore)
	ON_UPDATE_COMMAND_UI(ID_VIEW_EXPLORE, OnUpdateViewExplore)
	ON_COMMAND(ID_VIEW_IGNORE, OnViewIgnore)
	ON_UPDATE_COMMAND_UI(ID_VIEW_IGNORE, OnUpdateViewIgnore)
	ON_UPDATE_COMMAND_UI(ID_VIEW_DIFF, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_EDIT, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_EDITORS, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_LOCKF, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_LOG, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_GRAPH, OnUpdateViewRmv)
	ON_UPDATE_COMMAND_UI(ID_VIEW_TAGNEW, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_TAGDELETE, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_TAGBRANCH, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_STATUS, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_UNEDIT, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_UPDATE, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_UNLOCKF, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_WATCHERS, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_WATCHON, OnUpdateViewUpdate)
	ON_UPDATE_COMMAND_UI(ID_VIEW_WATCHOFF, OnUpdateViewUpdate)
	ON_WM_TIMER()
	//}}AFX_MSG_MAP
	ON_COMMAND_EX_RANGE(ID_MACRO_SEL, ID_MACRO_SEL + 99, OnMacroSel)
END_MESSAGE_MAP()

CImageList CBrowseFileView::m_SmallImageList;
CImageList CBrowseFileView::m_StateImageList;

class TViewFill : public TraversalReport
{
public:
	CListCtrl & m_listCtrl;
	int m_count;
	CSortList<ENTNODE> & m_entries;
	vector<CStr> m_ignlist;

	TViewFill(CListCtrl & listCtrl, CSortList<ENTNODE> & entries) :
		m_listCtrl(listCtrl), m_count(0), m_entries(entries) {}

	virtual kTraversal EnterDirectory(const char *fullpath, const char *dirname)
	{
		if(chdir("CVS") == 0)
		{
			Entries_Open (m_entries, fullpath);
			if(chdir(fullpath) != 0)
				return kTraversalError;
			BuildIgnoredList(m_ignlist);
		}
		return kContinueTraversal;
	}

	virtual kTraversal ExitDirectory(const char *fullpath)
	{
		m_ignlist.erase(m_ignlist.begin(), m_ignlist.end());
		return kContinueTraversal;
	}

	virtual kTraversal OnError(const char *err, int errcode)
	{
		return kTraversalError;
	}

	virtual kTraversal OnIdle(const char *fullpath)
	{
		return kContinueTraversal;
	}

	virtual kTraversal OnDirectory(const char *fullpath,
		const char *fullname,
		const char *name,
		const struct stat & dir, FSSpec * macspec)
	{
		if(stricmp(name, "CVS") == 0)
			return kSkipFile;

		EntnodeData *data = Entries_SetVisited(m_entries, name, dir, true, &m_ignlist);
		if(!(bool)gFileViewIgnore && data->IsIgnored())
			return kSkipFile;

		// get the tag
		CStr subCVS;
		CStr tagName;
		subCVS = fullname;
		if(!subCVS.endsWith(kPathDelimiter))
			subCVS << kPathDelimiter;
		subCVS << "CVS";
		if(chdir(subCVS) == 0)
			Tag_Open(tagName, subCVS);
		if(chdir(fullpath) != 0)
			return kTraversalError;

		LV_ITEM lvi;
		lvi.mask = LVIF_TEXT | LVIF_IMAGE;
		lvi.iItem = m_count;
		lvi.iSubItem = 0;
		lvi.pszText = (char *)name;
		lvi.iImage = CBrowseFileView::GetImageForEntry(data);

		m_listCtrl.InsertItem(&lvi);
		m_listCtrl.SetItemData(m_count, (DWORD)data);

		// set item text for additional columns
		for(int j = 1; j < NUM_COLUMNS; j++)
		{
			const char *info = (*data)[j];
			if(info != 0L)
				m_listCtrl.SetItemText(m_count, j, info);
		}
		m_listCtrl.SetItemText(m_count, EntnodeData::kStatus, data->GetDesc());
		m_listCtrl.SetItemText(m_count, EntnodeFile::kTag, tagName);
		m_count++;

		return kSkipFile;
	}

	virtual kTraversal OnAlias(const char *fullpath,
		const char *fullname,
		const char *name,
		const struct stat & dir, FSSpec * macspec)
	{
		return OnFile(fullpath, fullname, name, dir, macspec);
	}

	virtual kTraversal OnFile(const char *fullpath,
		const char *fullname,
		const char *name,
		const struct stat & dir, FSSpec * macspec)
	{
		EntnodeData *data = Entries_SetVisited(m_entries, name, dir, false, &m_ignlist);
		if(!(bool)gFileViewIgnore && data->IsIgnored())
			return kContinueTraversal;

		LV_ITEM lvi;
		lvi.mask = LVIF_TEXT | LVIF_IMAGE | LVIF_STATE;
		lvi.iItem = m_count;
		lvi.iSubItem = 0;
		lvi.pszText = (char *)name;
		lvi.iImage = CBrowseFileView::GetImageForEntry(data);
		lvi.stateMask = LVIS_STATEIMAGEMASK;
		lvi.state = data->IsLocked() ?
			INDEXTOSTATEIMAGEMASK(2) : INDEXTOSTATEIMAGEMASK(1);

		m_listCtrl.InsertItem(&lvi);
		m_listCtrl.SetItemData(m_count, (DWORD)data);

		// set item text for additional columns
		for(int j = 1; j < NUM_COLUMNS; j++)
		{
			const char *info = (*data)[j];
			if(info != 0L)
				m_listCtrl.SetItemText(m_count, j, info);
		}
		m_listCtrl.SetItemText(m_count, EntnodeData::kStatus, data->GetDesc());
		m_count++;

		return kContinueTraversal;
	}
};

/////////////////////////////////////////////////////////////////////////////
// CBrowseFileView construction/destruction

CBrowseFileView::CBrowseFileView() : m_entries(200, ENTNODE::Compare),
	m_checkChanges(false)
{
	m_sort = (int)gFileViewSort;
	m_ascendant = (bool)gFileViewSortAsc;
	m_entriesMod = 0;
	m_entriesLogMod = 0;
	m_watcher = 0L;

	if(!CWatcherThread::m_lockinit)
	{
		InitializeCriticalSection(&CWatcherThread::m_lock);
		CWatcherThread::m_lockinit = true;
	}
}

CBrowseFileView::~CBrowseFileView()
{
	gFileViewSort = m_sort;
	gFileViewSortAsc = m_ascendant;
	if(m_watcher != 0L)
	{
		HANDLE hdl = m_watcher->m_hThread;
		m_watcher->KillThread();
		WaitForSingleObject(hdl, INFINITE);
		//delete m_watcher;
		m_watcher = 0L;
	}
}

void CBrowseFileView::InvalidateView(void)
{
	m_entriesMod = 0;
}

BOOL CBrowseFileView::PreCreateWindow(CREATESTRUCT& cs)
{
	cs.style |= WS_TABSTOP | LVS_SHOWSELALWAYS | LVS_REPORT;
	cs.dwExStyle |= LVS_EX_FULLROWSELECT;

	return CListView::PreCreateWindow(cs);
}

int CBrowseFileView::GetImageForEntry(EntnodeData *data)
{
	int result;
	if(data->GetType() == ENT_FILE)
	{
		const char *info = 0L;
		if(data->IsIgnored())
		{
			result = kFileIconIgnored;
		}
		else if(data->IsUnknown())
		{
			result = kFileIconUnknown;
		}
		else if((*data)[EntnodeFile::kConflict] != 0L)
		{
			result = kFileIconConflict;
		}
		else if((info = (*data)[EntnodeFile::kOption]) != 0L && strcmp(info, "-kb") == 0)
		{
			result = data->IsUnmodified() ? kFileIconBinary : kFileIconBinaryMod;
		}
		else
		{
			result = data->IsUnmodified() ? kFileIconText : kFileIconTextMod;
		}
	}
	else
	{
		if(data->IsIgnored())
		{
			result = kFolderIconIgnored;
		}
		else if(data->IsUnknown())
		{
			result = kFolderIconUnknown;
		}
		else
		{
			result = kFolderIcon;
		}
	}

	return result;
}

/////////////////////////////////////////////////////////////////////////////
// CBrowseFileView initialization

void CBrowseFileView::OnInitialUpdate()
{
	CListView::OnInitialUpdate();

	CListCtrl& ListCtrl = GetListCtrl();

// set image lists
	if(m_SmallImageList.m_hImageList == 0L)
		m_SmallImageList.Create(IDB_SMALLICONS, 16, 1, RGB(255, 255, 255));
	if(m_StateImageList.m_hImageList == 0L)
		m_StateImageList.Create(IDR_STATEICONS, 16, 1, RGB(255, 0, 0));
	
	ListCtrl.SetImageList(&m_SmallImageList, LVSIL_SMALL);
	ListCtrl.SetImageList(&m_StateImageList, LVSIL_STATE);

	int i;
	LV_COLUMN lvc;

	lvc.mask = LVCF_FMT | LVCF_WIDTH | LVCF_TEXT | LVCF_SUBITEM;

	for(i = 0; i<NUM_COLUMNS; i++)
	{
		lvc.iSubItem = i;
		lvc.pszText = _gszColumnLabel[i];
		lvc.cx = _gnColumnWidth[i];
		lvc.fmt = _gnColumnFmt[i];
		ListCtrl.InsertColumn(i,&lvc);
	}

	SetFullRowSel(TRUE);
	SetTimer(999, 200, 0L);
}

void CBrowseFileView::GetEntriesModTime(time_t & newEntriesMod,
										time_t & newEntriesLogMod)
{
	newEntriesMod = 0;
	newEntriesLogMod = 0;
	if(chdir(m_path) != 0)
		return;
	if(chdir("CVS") != 0)
		return;

	struct stat sb;
	if (stat("Entries", &sb) != -1)
		newEntriesMod = sb.st_mtime;
	if (stat("Entries.log", &sb) != -1)
		newEntriesLogMod = sb.st_mtime;
	chdir(m_path);
}

class CFileWatcherLock
{
public:
	CFileWatcherLock()
	{
		EnterCriticalSection(&CWatcherThread::m_lock);
	}
	~CFileWatcherLock()
	{
		LeaveCriticalSection(&CWatcherThread::m_lock);
	}
};

CRITICAL_SECTION CWatcherThread::m_lock;
bool CWatcherThread::m_lockinit = false;

CWatcherThread::CWatcherThread() : m_watcher(INVALID_HANDLE_VALUE), m_term(false), m_Modified(false)
{
}

void CWatcherThread::Delete()
{
	if(m_watcher != INVALID_HANDLE_VALUE)
		FindCloseChangeNotification(m_watcher); 

	CWinThread::Delete();
}

void CWatcherThread::SetPath(const char *path)
{
	CFileWatcherLock locker;

	m_newpath = path;
	if (m_newpath.endsWith('\\'))	//Trim trailing backslash
		m_newpath[m_newpath.length()-1]='\0';
	m_Modified = false;
}

CWatcherThread::~CWatcherThread()
{
}

BOOL CWatcherThread::InitInstance()
{
	for(;;)
	{
		if(m_term)
			return FALSE;

		{
			CFileWatcherLock locker;

			if(strcmp(m_path, m_newpath) != 0 || m_path.empty())
			{
				if(m_watcher != INVALID_HANDLE_VALUE)
					FindCloseChangeNotification(m_watcher); 
				m_watcher = INVALID_HANDLE_VALUE;
				m_Modified = false;

				m_path = m_newpath;
			}
		}

		if(m_path.empty())
		{
			Sleep(200);
			continue;
		}

		if(m_watcher == INVALID_HANDLE_VALUE)
		{
			m_watcher = FindFirstChangeNotification( 
				m_path, 
				FALSE, 
				FILE_NOTIFY_CHANGE_FILE_NAME |
				FILE_NOTIFY_CHANGE_DIR_NAME |
				//FILE_NOTIFY_CHANGE_SIZE |
				FILE_NOTIFY_CHANGE_LAST_WRITE |
				//FILE_NOTIFY_CHANGE_LAST_ACCESS |
#if _MSC_VER >= 1200
				FILE_NOTIFY_CHANGE_CREATION |
#endif
				//FILE_NOTIFY_CHANGE_SECURITY |
				FILE_NOTIFY_CHANGE_ATTRIBUTES
			);
		}

		if(m_watcher == INVALID_HANDLE_VALUE)
		{
			Sleep(200);
			continue;
		}

		DWORD dw = WaitForSingleObject(m_watcher, 100);
 
        switch(dw) 
        { 
			case WAIT_OBJECT_0: 
				{
					m_Modified = true;
					BOOL res = FindNextChangeNotification(m_watcher);

					if(res == FALSE) 
					{ 
						FindCloseChangeNotification(m_watcher); 
						m_watcher = INVALID_HANDLE_VALUE;
						m_Modified = false;
						//continue;
					}
				}
				break;
			case WAIT_ABANDONED: 
				break; 
			case (WAIT_TIMEOUT): 
				break; 
			default: 
				break;
        }
	}

	return FALSE;
}

bool CWatcherThread::WasPathModified(bool waitABit) const
{
	if(m_Modified)
		return true;

	// wait a bit so we gime time for the above function
	// to set the m_Modified if necessary
	if(waitABit && m_watcher != INVALID_HANDLE_VALUE)
	{
		WaitForSingleObject(m_watcher, 200);
		if(m_Modified)
			return true;
	}

	return false;
}

IMPLEMENT_DYNAMIC(CWatcherThread, CWinThread);

BEGIN_MESSAGE_MAP(CWatcherThread, CWinThread)
	//{{AFX_MSG_MAP(CWatcherThread)
		// NOTE - the ClassWizard will add and remove mapping macros here.
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

#if qDebug
// to verify that nobody call ResetView twice : we don't want a critical section
// because we want ResetView only to be called from the main thread. The timer
// is here to call ResetView whenever another thread notify it.
static bool sDummySemaphoreDbg = false;
class CDummySemaphoreDbg
{
public:
	CDummySemaphoreDbg(bool & semaphore) : m_semaphore(semaphore), m_didSetSemaphore(false)
	{
		if(m_semaphore)
		{
			cvs_err("ResetView was called twice !\n");
			throw(EPERM);
		}
		if(GetCurrentThreadId() != AfxGetApp()->m_nThreadID)
		{
			cvs_err("ResetView should be called only by the main thread !\n");
			throw(EPERM);
		}
		m_semaphore = true;
		m_didSetSemaphore = true;
	}

	~CDummySemaphoreDbg()
	{
		if(m_didSetSemaphore)
			m_semaphore = false;
	}
protected:
	bool & m_semaphore;
	bool m_didSetSemaphore;
};
#endif

void CBrowseFileView::ResetView(bool forceReload, bool notifyBrowser)
{
#if qDebug
	CDummySemaphoreDbg policeman(sDummySemaphoreDbg);
#endif

	CWaitCursor doWait;

	// - check if we really need to reload (forceReload == false)
	// - wait a bit for the watcher in order to let him tell us
	// if something was modified.
	time_t newEntriesMod;
	time_t newEntriesLogMod;
	GetEntriesModTime(newEntriesMod, newEntriesLogMod);
	if(!forceReload)
	{
		if(m_entriesMod == newEntriesMod && m_entriesLogMod == newEntriesLogMod &&
			(m_watcher == 0L || !m_watcher->WasPathModified()))
				return;
	}

	// reset all the watchers
	m_entriesMod = newEntriesMod;
	m_entriesLogMod = newEntriesLogMod;
	if(m_watcher != 0L)
		m_watcher->ResetPathModified();

	CListCtrl& ListCtrl = GetListCtrl();
	int i;

	// get the selection to later try to restore it
	CvsArgs selection(false);
	int nItem = -1;

	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		selection.add((*data)[EntnodeData::kName]);
	}

	// delete all items
	VERIFY(ListCtrl.DeleteAllItems());
	m_entries.Reset();

	// refetch all items
	TViewFill traverse(ListCtrl, m_entries);
	kTraversal res = FileTraverse(m_path, traverse);

	// add the missing files
	Entries_SetMissing(m_entries);

	int numEntries = m_entries.NumOfElements();
	for(i = 0; i < numEntries; i++)
	{
		const ENTNODE & theNode = m_entries.Get(i);
		EntnodeData *data = ((ENTNODE *)&theNode)->Data();
		if(!data->IsMissing())
			continue;

		LV_ITEM lvi;
		lvi.mask = LVIF_TEXT | LVIF_IMAGE;
		lvi.iItem = traverse.m_count;
		lvi.iSubItem = 0;
		lvi.pszText = (char *)(*data)[EntnodeData::kName];
		lvi.iImage = data->GetType() == ENT_FILE ?
			kFileIconMiss : kFolderIconMiss;

		ListCtrl.InsertItem(&lvi);
		ListCtrl.SetItemData(traverse.m_count, (DWORD)data);

		// set item text for additional columns
		for(int j = 1; j < NUM_COLUMNS; j++)
		{
			const char *info = (*data)[j];
			if(info != 0L)
				ListCtrl.SetItemText(traverse.m_count, j, info);
		}
		ListCtrl.SetItemText(traverse.m_count, EntnodeData::kStatus, data->GetDesc());
		traverse.m_count++;
	}

	Resort();

	// now restore the selection
	int argc = selection.Argc();
	char * const *argv = selection.Argv();
	LV_FINDINFO FindInfo;
	for(i = 0; i < argc; i++)
	{
		FindInfo.flags = LVFI_STRING;
		FindInfo.psz = argv[i];
		nItem = ListCtrl.FindItem(&FindInfo, -1);
		if(nItem != -1)
		{
			VERIFY(ListCtrl.SetItemState(nItem, LVIS_SELECTED,
				LVIS_SELECTED));
			if(i == 0)
				ListCtrl.EnsureVisible(nItem, /*bPartialOK*/TRUE);

		}
	}

	if(notifyBrowser)
	{
		// notify the tree
		CWincvsApp* app = (CWincvsApp *)AfxGetApp();
		app->GetBrowserView()->ResetView(forceReload);
	}

	// create the watcher thread
	if(m_watcher == 0L)
	{
		m_watcher = new CWatcherThread();
		if (m_watcher == 0L)
		{
			cvs_err("Impossible to create the watcher thread !\n");
		}
		else
		{
			ASSERT_VALID(m_watcher);

			// Create Thread in a suspended state so we can set the Priority 
			// before it starts getting away from us
			if (!m_watcher->CreateThread(CREATE_SUSPENDED))
			{
				cvs_err("Impossible to start the watcher thread (error %d)!\n", GetLastError());
				delete m_watcher;
				m_watcher = 0L;
			}

			if(m_watcher != 0L)
			{
				// If you want to make the sample more sprightly, set the thread priority here 
				// a little higher. It has been set at idle priority to keep from bogging down 
				// other apps that may also be running.
				VERIFY(m_watcher->SetThreadPriority(THREAD_PRIORITY_NORMAL));
				// Now the thread can run wild
				m_watcher->ResumeThread();
			}
		}
	}

	if(m_watcher != 0L)
		m_watcher->SetPath(m_path);
}

void CBrowseFileView::ResetView(const char *path, bool notifyBrowser)
{
	m_path = path;
	ResetView(true);

	if(notifyBrowser)
	{
		// notify the tree
		CWincvsApp* app = (CWincvsApp *)AfxGetApp();
		app->GetBrowserView()->StepToLocation(path);
	}
}

/////////////////////////////////////////////////////////////////////////////
// CBrowseFileView diagnostics

#ifdef _DEBUG
void CBrowseFileView::AssertValid() const
{
	CListView::AssertValid();
}

void CBrowseFileView::Dump(CDumpContext& dc) const
{
	CListView::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CBrowseFileView helpers

BOOL CBrowseFileView::SetViewType(DWORD dwViewType)
{
	return(ModifyStyle(LVS_TYPEMASK,dwViewType & LVS_TYPEMASK));
}

DWORD CBrowseFileView::GetViewType()
{
	return(GetStyle() & LVS_TYPEMASK);
}

/////////////////////////////////////////////////////////////////////////////
// CBrowseFileView message handlers

void CBrowseFileView::OnViewSmallIcons() 
{
	if (GetViewType() != LVS_SMALLICON)
		SetViewType(LVS_SMALLICON);
}

void CBrowseFileView::OnViewList() 
{
	if (GetViewType() != LVS_LIST)
		SetViewType(LVS_LIST);
}

void CBrowseFileView::OnViewFullRowDetails() 
{
	if (GetViewType() != LVS_REPORT)
		SetViewType(LVS_REPORT);
}

void CBrowseFileView::OnUpdateViewSmallIcons(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(GetViewType() == LVS_SMALLICON);
}

void CBrowseFileView::OnUpdateViewList(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(GetViewType() == LVS_LIST);
}

void CBrowseFileView::OnUpdateViewFullRowDetails(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(GetViewType() == LVS_REPORT);
}

void CBrowseFileView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	CListView::OnLButtonDown(nFlags, point);
}

void CBrowseFileView::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	if(app->gCvsRunning)
		return;

	CListCtrl& ListCtrl = GetListCtrl();

	UINT pFlags;
	int nItem = ListCtrl.HitTest(point, &pFlags);
	if(nItem != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		CStr fullpath;
		fullpath = m_path;
		if(!fullpath.endsWith(kPathDelimiter))
			fullpath << kPathDelimiter;
		fullpath << (*data)[EntnodeData::kName];
		if(data->GetType() == ENT_SUBDIR)
		{
			ResetView(fullpath, true);
		}
		else
		{
			char viewer[_MAX_PATH];
			HINSTANCE hInst = FindExecutable((*data)[EntnodeData::kName],
				m_path, viewer);
			if((UINT)hInst >= 32)
			{
				ShellExecute(*AfxGetMainWnd(), "open", (*data)[EntnodeData::kName],
					NULL, m_path, SW_SHOWDEFAULT);
			} 
			else
			{
				const char * argv[3] = {0L, 0L, 0L};
				CStr program, file;
				if(strchr(gCvsPrefs.Viewer(), ' ') != 0L)
				{
					program << '\"';
					program << gCvsPrefs.Viewer();
					program << '\"';
				}
				else
					program = gCvsPrefs.Viewer();
				if(strchr(fullpath, ' ') != 0L)
				{
					file << '\"';
					file << fullpath;
					file << '\"';
				}
				else
					file = fullpath;
				argv[0] = program;
				argv[1] = file;
				int process = _spawnvp(_P_NOWAIT, gCvsPrefs.Viewer(), argv);
			}
		}
	}

	CListView::OnLButtonDblClk(nFlags, point);
}

void CBrowseFileView::OnUpdateViewRelease(CCmdUI* pCmdUI)
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	CListCtrl& ListCtrl = GetListCtrl();

	if(ListCtrl.GetSelectedCount() == 0 || app->gCvsRunning || gCvsPrefs.empty())
	{
		pCmdUI->Enable(FALSE);
		return;
	}

	int nItem = -1;
	BOOL res = TRUE;
	int numFiles = 0;
	int numFolders = 0;

	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->IsUnknown() || data->GetType() != ENT_SUBDIR)
		{
			res = FALSE;
			break;
		}
		numFolders++;

		if(numFolders > 1)
		{
			res = FALSE;
			break;
		}
	}

	pCmdUI->Enable(res);
}

void CBrowseFileView::OnUpdateViewAdd(CCmdUI* pCmdUI)
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	CListCtrl& ListCtrl = GetListCtrl();

	if(ListCtrl.GetSelectedCount() == 0 || app->gCvsRunning || gCvsPrefs.empty())
	{
		pCmdUI->Enable(FALSE);
		return;
	}

	int nItem = -1;
	BOOL res = TRUE;
	int numFiles = 0;
	int numFolders = 0;

	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(!data->IsUnknown())
		{
			res = FALSE;
			break;
		}
		if(data->GetType() == ENT_FILE)
			numFiles++;
		else
			numFolders++;

		if((numFiles != 0 && numFolders != 0) || numFolders > 1)
		{
			res = FALSE;
			break;
		}
	}

	pCmdUI->Enable(res);
}

void CBrowseFileView::OnUpdateViewAddB(CCmdUI* pCmdUI)
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	CListCtrl& ListCtrl = GetListCtrl();

	if(ListCtrl.GetSelectedCount() == 0 || app->gCvsRunning || gCvsPrefs.empty())
	{
		pCmdUI->Enable(FALSE);
		return;
	}

	int nItem = -1;
	BOOL res = TRUE;

	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(!data->IsUnknown() || data->GetType() != ENT_FILE)
		{
			res = FALSE;
			break;
		}
	}

	pCmdUI->Enable(res);
}

void CBrowseFileView::OnUpdateViewUpdate(CCmdUI* pCmdUI)
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	CListCtrl& ListCtrl = GetListCtrl();

	if(ListCtrl.GetSelectedCount() == 0 || app->gCvsRunning || gCvsPrefs.empty())
	{
		pCmdUI->Enable(FALSE);
		return;
	}

	int nItem = -1;
	BOOL res = TRUE;
	int numFiles = 0;
	int numFolders = 0;

	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->IsUnknown())
		{
			res = FALSE;
			break;
		}
		if(data->GetType() == ENT_FILE)
			numFiles++;
		else
			numFolders++;

		if((numFiles != 0 && numFolders != 0) || numFolders > 1)
		{
			res = FALSE;
			break;
		}
	}

	pCmdUI->Enable(res);
}

void CBrowseFileView::OnUpdateViewRmv(CCmdUI* pCmdUI)
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	CListCtrl& ListCtrl = GetListCtrl();

	if(ListCtrl.GetSelectedCount() == 0 || app->gCvsRunning || gCvsPrefs.empty())
	{
		pCmdUI->Enable(FALSE);
		return;
	}

	int nItem = -1;
	BOOL res = TRUE;

	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->IsUnknown() || data->GetType() == ENT_SUBDIR)
		{
			res = FALSE;
			break;
		}
	}

	pCmdUI->Enable(res);
}

static int CALLBACK compareName(LPARAM data1, LPARAM data2, LPARAM data)
{
	EntnodeData *d1 = (EntnodeData *)data1;
	EntnodeData *d2 = (EntnodeData *)data2;
	int res = stricmp((*d1)[EntnodeData::kName], (*d2)[EntnodeData::kName]);
	return data ? res : -res;
}

static int CALLBACK compareStatus(LPARAM data1, LPARAM data2, LPARAM data)
{
	EntnodeData *d1 = (EntnodeData *)data1;
	EntnodeData *d2 = (EntnodeData *)data2;
	int res = stricmp((*d1)[EntnodeData::kStatus], (*d2)[EntnodeData::kStatus]);
	if(res == 0)
		res = stricmp((*d1)[EntnodeData::kName], (*d2)[EntnodeData::kName]);
	return data ? res : -res;
}

static int CALLBACK compareOption(LPARAM data1, LPARAM data2, LPARAM data)
{
	EntnodeData *d1 = (EntnodeData *)data1;
	EntnodeData *d2 = (EntnodeData *)data2;
	const char *s1 = (*d1)[EntnodeFile::kOption];
	const char *s2 = (*d2)[EntnodeFile::kOption];
	int res;
	if(s1 != 0L && s2 != 0L)
		res = strcmp(s1, s2);
	else
		res = (long)s1 < (long)s2 ? -1 : ((long)s1 > (long)s2 ? 1 : 0);
	if(res == 0)
		res = stricmp((*d1)[EntnodeData::kName], (*d2)[EntnodeData::kName]);
	return data ? res : -res;
}

static int CALLBACK compareTag(LPARAM data1, LPARAM data2, LPARAM data)
{
	EntnodeData *d1 = (EntnodeData *)data1;
	EntnodeData *d2 = (EntnodeData *)data2;
	const char *s1 = (*d1)[EntnodeFile::kTag];
	const char *s2 = (*d2)[EntnodeFile::kTag];
	int res;
	if(s1 != 0L && s2 != 0L)
		res = strcmp(s1, s2);
	else
		res = (long)s1 < (long)s2 ? -1 : ((long)s1 > (long)s2 ? 1 : 0);
	if(res == 0)
		res = stricmp((*d1)[EntnodeData::kName], (*d2)[EntnodeData::kName]);
	return data ? res : -res;
}

static int CALLBACK compareConflict(LPARAM data1, LPARAM data2, LPARAM data)
{
	EntnodeData *d1 = (EntnodeData *)data1;
	EntnodeData *d2 = (EntnodeData *)data2;
	const char *s1 = (*d1)[EntnodeFile::kConflict];
	const char *s2 = (*d2)[EntnodeFile::kConflict];
	int res;
	if(s1 != 0L && s2 != 0L)
		res = strcmp(s1, s2);
	else
		res = (long)s1 < (long)s2 ? -1 : ((long)s1 > (long)s2 ? 1 : 0);
	if(res == 0)
		res = stricmp((*d1)[EntnodeData::kName], (*d2)[EntnodeData::kName]);
	return data ? res : -res;
}

void CBrowseFileView::Resort(void)
{
	CListCtrl& ListCtrl = GetListCtrl();
	switch(m_sort)
	{
	case EntnodeData::kName:
		ListCtrl.SortItems(compareName, m_ascendant);
		break;
	case EntnodeData::kStatus:
		ListCtrl.SortItems(compareStatus, m_ascendant);
		break;
	case EntnodeFile::kOption:
		ListCtrl.SortItems(compareOption, m_ascendant);
		break;
	case EntnodeFile::kTag:
		ListCtrl.SortItems(compareTag, m_ascendant);
		break;
	case EntnodeFile::kConflict:
		ListCtrl.SortItems(compareConflict, m_ascendant);
		break;
	}
}

void CBrowseFileView::OnColumnclick(NMHDR* pNMHDR, LRESULT* pResult) 
{
	NM_LISTVIEW* pNMListView = (NM_LISTVIEW*)pNMHDR;
	CListCtrl& ListCtrl = GetListCtrl();

	if(pNMListView->iSubItem == m_sort)
		m_ascendant = !m_ascendant;
	else
	{
		m_ascendant = true;
		switch(pNMListView->iSubItem)
		{
		case EntnodeData::kName:
		case EntnodeData::kStatus:
		case EntnodeFile::kOption:
		case EntnodeFile::kTag:
		case EntnodeFile::kConflict:
			m_sort = pNMListView->iSubItem;
			break;
		default:
			*pResult = 1;
			return;
		}
	}

	Resort();
	*pResult = 0;
}

void CBrowseFileView::OnViewAdd() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdAddFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdAddFiles(&mf);
	}
}

void CBrowseFileView::OnViewAddb() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdAddFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdAddBinaryFiles(&mf);
	}
}

void CBrowseFileView::OnViewCommit() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdCommitFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdCommitFiles(&mf);
	}
}

void CBrowseFileView::OnViewRmv() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		mf.newfile((*data)[EntnodeData::kName]);
	}

	if(mf.NumFiles() != 0)
	{
		CvsCmdRemoveFiles(&mf);
	}
}

void CBrowseFileView::OnViewUpdate() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdUpdateFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdUpdateFiles(&mf);
	}
}

void CBrowseFileView::OnUpdateViewReload(CCmdUI* pCmdUI) 
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	pCmdUI->Enable(!app->gCvsRunning);
}

void CBrowseFileView::OnViewReload() 
{
	ResetView(true, true);
}

void CBrowseFileView::OnKeydown(NMHDR* pNMHDR, LRESULT* pResult) 
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	if(app->gCvsRunning)
		return;

	CListCtrl& ListCtrl = GetListCtrl();
	LV_KEYDOWN* pLVKeyDow = (LV_KEYDOWN*)pNMHDR;

	if(pLVKeyDow->wVKey == VK_BACK)
	{
		OnViewUpone();
	}
	else if(pLVKeyDow->wVKey == VK_RETURN && ListCtrl.GetSelectedCount() == 1)
	{
		int nItem = -1;
		if((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
		{
			EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
			if(data->GetType() == ENT_SUBDIR)
			{
				CStr fullpath;
				fullpath = m_path;
				if(!fullpath.endsWith(kPathDelimiter))
					fullpath << kPathDelimiter;
				fullpath << (*data)[EntnodeData::kName];
				ResetView(fullpath, true);
			}
		}
	}
	
	*pResult = 0;
}

void CBrowseFileView::OnViewUpone() 
{
	if(chdir(m_path) == 0 && chdir("..") == 0)
	{
		char newpath[1024];
		getcwd(newpath, 1023);
		ResetView(newpath, true);
	}
}

void CBrowseFileView::OnUpdateViewUpone(CCmdUI* pCmdUI) 
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	pCmdUI->Enable(!app->gCvsRunning);
}

void CBrowseFileView::OnViewTrash() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		mf.newfile((*data)[EntnodeData::kName]);
	}

	if(mf.NumFiles() != 0)
	{
		CvsArgs args(false);
		mf.next();
		const char *dir = mf.add(args);
		if(chdir(dir) == 0)
		{
			char * const *argv = args.Argv();
			int argc = args.Argc();
			for(int i = 0; i < argc; i++)
			{
				if(chmod(argv[i], 0666) != 0)
				{
					cvs_err("Unable to change permission on '%s' (error %d)\n", argv[i], errno);
					continue;
				}

				if(!CompatMoveToTrash(argv[i], dir))
					cvs_err("Unable to remove '%s' (error %d)\n", argv[i], errno);
				else
					cvs_out("'%s' has been moved successfully to the recycle bin...\n", argv[i]);
			}
		}
		else
			cvs_err("Unable to access '%s' (error %d)\n", dir, errno);

		ResetView(true);
	}
}

void CBrowseFileView::OnUpdateViewTrash(CCmdUI* pCmdUI) 
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	CListCtrl& ListCtrl = GetListCtrl();
	if(ListCtrl.GetSelectedCount() == 0 || app->gCvsRunning || gCvsPrefs.empty())
	{
		pCmdUI->Enable(FALSE);
		return;
	}

	int nItem = -1;
	BOOL res = TRUE;

	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR || data->IsMissing())
		{
			res = FALSE;
			break;
		}
	}

	pCmdUI->Enable(res);
}

void CBrowseFileView::OnViewDiff() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdDiffFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdDiffFiles(&mf);
	}
}

void CBrowseFileView::OnViewLog() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdLogFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdLogFiles(&mf);
	}
}

void CBrowseFileView::OnViewGraph() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
			continue;

		mf.newfile((*data)[EntnodeData::kName]);
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdLogFiles(&mf, true);
	}
}

void CBrowseFileView::OnViewStatus() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdStatusFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdStatusFiles(&mf);
	}
}

void CBrowseFileView::OnViewLock() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdLockFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdLockFiles(&mf);
	}
}

void CBrowseFileView::OnViewUnlock() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdUnlockFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdUnlockFiles(&mf);
	}
}

void CBrowseFileView::OnViewWatchOn() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdWatchOnFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdWatchOnFiles(&mf);
	}
}

void CBrowseFileView::OnViewWatchOff()
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdWatchOffFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdWatchOffFiles(&mf);
	}
}

void CBrowseFileView::OnViewEdit()
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdEditFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdEditFiles(&mf);
	}
}

void CBrowseFileView::OnViewUnedit()
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdUneditFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdUneditFiles(&mf);
	}
}

void CBrowseFileView::OnViewRelease()
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdReleaseFolder(fullpath);
		}
	}
}

void CBrowseFileView::OnViewWatchers()
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdWatchersFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdWatchersFiles(&mf);
	}
}

void CBrowseFileView::OnViewEditors()
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdEditorsFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdEditorsFiles(&mf);
	}
}

void CBrowseFileView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	CListView::OnRButtonDown(nFlags, point);

	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	CListCtrl& ListCtrl = GetListCtrl();

	if(ListCtrl.GetSelectedCount() == 0 || app->gCvsRunning || gCvsPrefs.empty())
		return;

	UINT pFlags;
	int testItem = ListCtrl.HitTest(point, &pFlags);
	if(testItem == -1)
		return;

	if(ListCtrl.GetItemState(testItem, LVIS_SELECTED) == 0)
		return;

	if(pFlags & (LVHT_ONITEMICON|LVHT_ONITEMLABEL|LVHT_ONITEMSTATEICON) == 0)
		return;

	CMenu* pMyMenu=app->GetSubCMenu(ID_VIEW_UPDATE,this);
	if (pMyMenu)
	{
		ClientToScreen(&point);
		pMyMenu->TrackPopupMenu(TPM_LEFTALIGN|TPM_LEFTBUTTON|TPM_RIGHTBUTTON, point.x, point.y, this);
	}
}

void CBrowseFileView::OnViewTagNew() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdTagCreateFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdTagCreateFiles(&mf);
	}
}

void CBrowseFileView::OnViewTagDelete() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdTagDeleteFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdTagDeleteFiles(&mf);
	}
}

void CBrowseFileView::OnViewTagBranch() 
{
	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	MultiFiles mf;
	mf.newdir(m_path);

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			CStr fullpath;
			fullpath = m_path;
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			CvsCmdTagBranchFolder(fullpath);
		}
		else
		{
			mf.newfile((*data)[EntnodeData::kName]);
		}
	}
	if(mf.NumFiles() != 0)
	{
		CvsCmdTagBranchFiles(&mf);
	}
}

BOOL CBrowseFileView::OnMacroSel(UINT nID)
{
	CTcl_Interp interp;
	CMacroEntry & entry = gMacrosSel.entries[nID - ID_MACRO_SEL];
	CStr path = entry.path;
	CTcl_Interp::Unixfy(path);

	TclBrowserReset();

	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		TclBrowserAppend(m_path, data);
	}

	interp.DoScriptVar("source \"%s\"", (const char *)path);

	return 1;
}

void CBrowseFileView::OnUpdateMacroSel(CCmdUI* pCmdUI) 
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	CListCtrl& ListCtrl = GetListCtrl();

	if(ListCtrl.GetSelectedCount() == 0 || app->gCvsRunning ||
		gCvsPrefs.empty() || !CTcl_Interp::IsAvail())
	{
		pCmdUI->Enable(FALSE);
		return;
	}

	if(pCmdUI->m_pSubMenu != 0L || pCmdUI->m_pMenu == 0L)
		return;

	int iMRUMax = pCmdUI->m_pMenu->GetMenuItemCount();
	for (int iMRU = 0; iMRU < iMRUMax; iMRU++)
		pCmdUI->m_pMenu->DeleteMenu(pCmdUI->m_nID + iMRU, MF_BYCOMMAND);

	vector<CMacroEntry>::const_iterator i;
	for(i = gMacrosSel.entries.begin(); i != gMacrosSel.entries.end(); ++i)
	{
		pCmdUI->m_pMenu->InsertMenu(pCmdUI->m_nIndex++,
			MF_STRING | MF_BYPOSITION, pCmdUI->m_nID++,
			(*i).name);
	}

	pCmdUI->m_nIndexMax = pCmdUI->m_pMenu->GetMenuItemCount();
	pCmdUI->m_nIndex--; // point to last menu added
	pCmdUI->m_bEnableChanged = TRUE;    // all the added items are enabled
}

void CBrowseFileView::OnViewExplore() 
{
	CStr fullpath(m_path);

	CListCtrl& ListCtrl = GetListCtrl();
	int nItem = -1;

	// first add the folders
	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->GetType() == ENT_SUBDIR)
		{
			if(!fullpath.endsWith(kPathDelimiter))
				fullpath << kPathDelimiter;
			fullpath << (*data)[EntnodeData::kName];
			break;
		}
	}

	HINSTANCE hInst = ShellExecute(*AfxGetMainWnd(), "explore", fullpath,
			0L, 0L, SW_SHOWDEFAULT);
	if((long)hInst < 32)
	{
		cvs_err("Unable to explore '%s' (error %d)\n", (char *)m_path, GetLastError());
	}
}

void CBrowseFileView::OnUpdateViewExplore(CCmdUI* pCmdUI) 
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	CListCtrl& ListCtrl = GetListCtrl();
	if(ListCtrl.GetSelectedCount() == 0 || app->gCvsRunning || gCvsPrefs.empty())
	{
		pCmdUI->Enable(FALSE);
		return;
	}

	int nItem = -1;
	int numItem = 0;
	BOOL res = TRUE;

	while((nItem = ListCtrl.GetNextItem(nItem, LVNI_SELECTED)) != -1)
	{
		EntnodeData *data = (EntnodeData *)ListCtrl.GetItemData(nItem);
		if(data->IsMissing() || ++numItem > 1)
		{
			res = FALSE;
			break;
		}
	}

	pCmdUI->Enable(res);
}

void CBrowseFileView::OnViewIgnore() 
{
	gFileViewIgnore = !(bool)gFileViewIgnore;
	ResetView(true, true);
}

void CBrowseFileView::OnUpdateViewIgnore(CCmdUI* pCmdUI) 
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	if(app->gCvsRunning || gCvsPrefs.empty())
	{
		pCmdUI->Enable(FALSE);
		return;
	}

	pCmdUI->SetCheck(gFileViewIgnore == true ? TRUE : FALSE);
}

void CBrowseFileView::OnTimer(UINT nIDEvent) 
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	if(!app->gCvsRunning)
	{
		if((m_watcher != 0L && m_watcher->WasPathModified(false)) || m_checkChanges)
		{
			m_checkChanges = false;
			ResetView();
		}
	}

	//CListView::OnTimer(nIDEvent);
}

void CBrowseFileView::SetFullRowSel(BOOL bFullRowSel)
{
	CListCtrl& ListCtrl = GetListCtrl();
	DWORD dwExStyle = ListView_GetExtendedListViewStyle(ListCtrl.m_hWnd);
	if (bFullRowSel)
		(dwExStyle |= LVS_EX_FULLROWSELECT);
	else
		(dwExStyle &= ~LVS_EX_FULLROWSELECT);
	ListView_SetExtendedListViewStyle(ListCtrl.m_hWnd, dwExStyle);
}
