/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- December 1997
 */

/*
 * AppGlue.cpp --- glue code to access CVS shared library
 */

#include "stdafx.h"

#ifdef macintosh
#	include <TextUtils.h>
#	include <Strings.h>
#	include <Dialogs.h>
#	include <unistd.h>
#	include <errno.h>
#	include <console.h>
#	include "mktemp.h"
#endif /* macintosh */

#ifdef qMacCvsPP
#	include "MacCvsApp.h"
#	include "LogWindow.h"
#	include "MacBinEncoding.h"
#	include "MacCvsAE.h"
#endif /* qMacCvsPP */

#ifdef WIN32
#	include "wincvs.h"
#	include "wincvsView.h"
#	include "PromptFiles.h"
#	include "BrowseFileView.h"
#	include <process.h>
#	include <direct.h>

#	ifdef _DEBUG
#	define new DEBUG_NEW
#	undef THIS_FILE
	static char THIS_FILE[] = __FILE__;
#	endif
#endif /* WIN32 */

#ifdef qQT
#	include "qcvsapp.h"
#	include "qcvsconsole.h"
#	include "qfileview.h"
#endif /* qQT */

#ifdef qQT
#	include <fcntl.h>
#	ifdef HAVE_UNISTD_H
#	include <unistd.h>
#	endif
	static long outseek;
	static long errseek;
	static char *outname;
	static char *errname;
	static int outlog = -1;
	static int errlog = -1;
	static int oldout = -1;
	static int olderr = -1;
	static int oldstdout = -1;
	static int oldstderr = -1;
#	ifdef FD_SETTER
#		define SET_FILE_FD_FIELD(F,D) ((F)->FD_SETTER = (D))
#		define GET_FILE_FD_FIELD(F) ((F)->FD_SETTER)
#	else
#		error "Don't know how to set the filoe descriptor"
#	endif
#endif /* qQT */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#ifdef HAVE_ERRNO_H
#	include <errno.h>
#endif
#include <vector>

#include "Authen.h"
#include "CvsPrefs.h"
#include "AppGlue.h"
#include "AppConsole.h"
#include "GetPassword.h"
#include "AskYesNo.h"
#include "CvsArgs.h"
#include "FileTraversal.h"

#include "dll_process.h"
#include "dll_loader.h"

// overide console
static CCvsConsole * sConsole = 0L;

#ifdef WIN32
	static CWinThread *sLastWin32CvsThread = 0L;
#endif

extern "C" {
	static long consoleout(char *txt, long len);
	static long consoleerr(char *txt, long len);
	static long consolein(char *txt, long len);
	static char *mygetenv(char *name);
#ifdef macintosh
	static int myeventdispatcher(EventRecord *ev);
	static int myeventhandler(int /*spinmsg*/, long /*arg*/);
#endif /* macintosh */
}

// the UNIX console handling to get stderr/stdout from cvs
// and output it on our console.
#ifdef qQT
static bool dupConsoles(void)
{
	oldstdout = GET_FILE_FD_FIELD(stdout);
	oldstderr = GET_FILE_FD_FIELD(stderr);

	outname = tempnam(0L, 0L);
	if(outname == 0L)
	{
		cvs_err("Unable to create a temp file (error %d)!\n", errno);
		return false;
	}
	errname = tempnam(0L, 0L);
	if(errname == 0L)
	{
		cvs_err("Unable to create a temp file (error %d)!\n", errno);
		return false;
	}
	oldout = dup(1);
	if(oldout == -1)
	{
		cvs_err("Unable to dup stdout (error %d)!\n", errno);
		return false;
	}
	outlog = open(outname, O_RDWR | O_CREAT | O_TRUNC/* | O_RANDOM*/, 0666);
	if(outlog == -1 || dup2(outlog, 1) == -1)
	{
		cvs_err("Unable to open stdout (error %d)!\n", errno);
		return false;
	}
	olderr = dup(2);
	if(olderr == -1)
	{
		cvs_err("Unable to dup stderr (error %d)!\n", errno);
		return false;
	}
	errlog = open(errname, O_RDWR | O_CREAT | O_TRUNC/* | O_RANDOM*/, 0666);
	if(errlog == -1 && dup2(errlog, 2) == -1)
	{
		cvs_err("Unable to open stderr (error %d)!\n", errno);
		return false;
	}
	SET_FILE_FD_FIELD(stdout, outlog);
	SET_FILE_FD_FIELD(stderr, errlog);
	outseek = 0;
	errseek = 0;
	return true;
}

static void restoreConsoles(void)
{
	SET_FILE_FD_FIELD(stdout, oldstdout);
	SET_FILE_FD_FIELD(stderr, oldstderr);

	if(oldout != -1)
	{
		dup2(oldout, 1);
		close(oldout);
		oldout = -1;
	}
	if(olderr != -1)
	{
		dup2(olderr, 2);
		close(olderr);
		olderr = -1;
	}
	if(outlog != -1)
	{
		close(outlog);
		outlog = -1;
	}
	if(errlog != -1)
	{
		close(errlog);
		errlog = -1;
	}
	if(outname != 0L)
	{
		unlink(outname);
		outname = 0L;
	}
	if(errname != 0L)
	{
		unlink(errname);
		errname = 0L;
	}
}

static void myflush(int log, long *oldpos)
{
	long pos;
#	define BUF_SIZE 8000
	char buf[BUF_SIZE];
	size_t len;

	pos = lseek(log, *oldpos, SEEK_SET);
	if(pos < 0)
		goto fail;

	while(1)
	{
		len = read(log, buf, sizeof(char) * BUF_SIZE);
		if(len > 0)
		{
			if(log == outlog)
				consoleout(buf, len);
			if(log == errlog)
				consoleerr(buf, len);
			pos += len;
		}
		else
			break;
	}

	*oldpos = pos;
	return;
fail:
	cvs_err("Problem while redirecting stdout/stderr (error %d)!\n", errno);
}

static void flushConsoles(void)
{
	if(outlog != -1)
		myflush(outlog, &outseek);
	if(errlog != -1)
		myflush(errlog, &errseek);
}
#endif /* qQT */

#ifdef _MSC_VER
extern "C"
#endif
static long consoleout(char *txt, long len)
{
	if(sConsole != 0L)
		return sConsole->cvs_out(txt, len);
	
#ifdef macintosh
	// apple event output
	long newlen;
	if((newlen = ae_consoleout(txt, len)) != -1)
		return newlen;
	
#	ifdef qMacCvsPP
	CLogWindow *log = CMacCvsApp::app->GetLogWindow();
	if(log != 0L)
		log->WriteCharsToConsole(txt, len);
	return len;
#	else /* !qMacCvsPP */
	return WriteCharsToConsole (txt, len);
#	endif /* !qMacCvsPP */
#endif /* macintosh */
#ifdef WIN32
	CWinApp* app = AfxGetApp();
	//ASSERT(app->IsKindOf(RUNTIME_CLASS(CWincvsApp)));
	CWincvsView *view = ((CWincvsApp *)app)->GetConsoleView();
	if(view == NULL)
		return 0;

	view->OutConsole(txt, len);

	return len;
#endif /* WIN32 */
#ifdef qQT
	QCvsConsole *console = QCvsApp::qApp->GetConsole();
	if(console == 0L)
		return 0;

	console->OutConsole(txt, len);

	return len;
#endif /* qQT */
}

#ifdef _MSC_VER
extern "C"
#endif
static long consoleerr(char *txt, long len)
{
	if(sConsole != 0L)
		return sConsole->cvs_err(txt, len);
	
#ifdef macintosh
	// apple event output
	long newlen;
	if((newlen = ae_consoleout(txt, len)) != -1)
		return newlen;

#	ifdef qMacCvsPP
	CLogWindow *log = CMacCvsApp::app->GetLogWindow();
	if(log != 0L)
		log->WriteCharsToConsole(txt, len, true);
	return len;
#	else /* !qMacCvsPP */
	return WriteCharsToConsole (txt, len);
#	endif /* !qMacCvsPP */
#endif /* macintosh */
#ifdef WIN32
	CWinApp* app = AfxGetApp();
	//ASSERT(app->IsKindOf(RUNTIME_CLASS(CWincvsApp)));
	CWincvsView *view = ((CWincvsApp *)app)->GetConsoleView();
	if(view == NULL)
		return 0;

	view->OutConsole(txt, len, true);

	return len;
#endif /* WIN32 */
#ifdef qQT
	QCvsConsole *console = QCvsApp::qApp->GetConsole();
	if(console == 0L)
		return 0;

	console->OutConsole(txt, len, true);

	return len;
#endif /* qQT */
}

#ifdef _MSC_VER
extern "C"
#endif
static long consolein(char *txt, long len)
{
#ifdef macintosh
	// apple event input
	long newlen;
	if((newlen = ae_consolein(txt, len)) != -1)
		return newlen;

	return 0;
#else /* !macintosh */
	return 0;
#endif /* !macintosh */
}

#ifdef macintosh
extern "C" static int myeventdispatcher(EventRecord *ev)
{
	// for 3.0 we could call SiouxHandleOneEvent
#	ifdef qMacCvsPP
	if(ev->what == mouseDown)
	{
		WindowPtr macWindowP;
		Int16 thePart = ::FindWindow(ev->where, &macWindowP);
		switch (thePart)
		{
			case inDrag:
			case inGrow:
			case inZoomIn:
			case inZoomOut:
				CMacCvsApp::app->DispatchEvent(*ev);
				break;
			case inContent:
				if (macWindowP != ::FrontWindow())
				{
					// Clicked Window is not the front one
					LWindow *frontW = LWindow::FetchWindowObject(macWindowP);
					if (frontW != nil && !UDesktop::WindowIsSelected(frontW))
						CMacCvsApp::app->DispatchEvent(*ev);
				}
				break;
		}
		return 0;
	}
	CMacCvsApp::app->DispatchEvent(*ev);
#	endif /* qMacCvsPP */
	return 0;
}
#endif /* macintosh */

#ifdef macintosh
#	define isGlobalEnvVariable(name) ( \
		strcmp(name, "CVS_PSERVER_PORT") == 0 || \
		strcmp(name, "CVS_RCMD_PORT") == 0 || \
		strcmp(name, "CVS_SERVER") == 0 || \
		strcmp(name, "CVS_RSH") == 0 || \
		strcmp(name, "HOME") == 0 || \
		strcmp(name, "ISO8859") == 0 || \
		strcmp(name, "IC_ON_TXT") == 0 || \
		strcmp(name, "MAC_DEFAULT_RESOURCE_ENCODING") == 0 || \
		strcmp(name, "MAC_BINARY_TYPES_SINGLE") == 0 || \
		strcmp(name, "MAC_BINARY_TYPES_HQX") == 0 || \
		strcmp(name, "MAC_BINARY_TYPES_PLAIN") == 0 || \
		strcmp(name, "DIRTY_MCP") == 0 \
		)
#endif /* macintosh */

#ifdef _MSC_VER
extern "C"
#endif
static char *mygetenv(char *name)
{
#ifdef WIN32
	// synchonize it since the thread calls and
	// we call some MFC objects in it (prompt
	// password, get path...)
	CCvsThreadLock();
#endif /* WIN32 */

	static char port[10];

	if(name == NULL)
		return NULL;
		
#ifdef macintosh
	char *result;
	if((result = ae_getenv(name)) != (char *)-1)
	{
		if(result == 0L && isGlobalEnvVariable(name))
		{
			// in this case only we let continue
			// to get the MacCVS settings. So the CWCVS
			// users (or any AppleEvent user) can set
			// these variables in MacCVS but still can overide
			// them using Apple Events.
		}
		else
			return result;
	}
#endif /* macintosh */
	
	if(strcmp(name, "USER") == 0
#ifdef WIN32
		// I do that (and see the "STRATA_HACK" in
		// subr.c) because people get tired on Windows
		// to have to check their Windows login :
		// this is only for .rhosts authentication.
		|| strcmp(name, "LOGNAME") == 0
#endif /* WIN32 */
		)
	{
		static char user[64];
		
		// extract from the cvsroot
		const char *ccvsroot = gCvsPrefs;
		ccvsroot = Authen::skiptoken(ccvsroot);
		char *login = strchr(ccvsroot, '@');
		if(login == NULL)
		{
			// for WIN32 this means the CVSROOT is local
#ifdef macintosh
			cvs_err("MacCvs: Cannot extract login from cvsroot %s !\n", ccvsroot);
#endif /* macintosh */
			return NULL;
		}
		int length = login - ccvsroot;
		memcpy(user, ccvsroot, length * sizeof(char));
		user[length] = '\0';
		return user;
	}
	else if(strcmp(name, "CVSROOT") == 0)
	{
		static CStr root;
		
		// extract from the cvsroot
		const char *ccvsroot = gCvsPrefs;
		ccvsroot = Authen::skiptoken(ccvsroot);
		root = gAuthen.token();
		root << ccvsroot;
		return root;
	}
	else if(strcmp(name, "CVSREAD") == 0)
	{
		return gCvsPrefs.CheckoutRO() ? "1" : 0L;
	}
	else if(strcmp(name, "CVS_GETPASS") == 0)
	{
		return CompatGetPassword();
	}
	else if(strcmp(name, "CVS_PSERVER_PORT") == 0)
	{
		if(gCvsPrefs.PserverPort() != -1)
		{
			sprintf(port, "%d", gCvsPrefs.PserverPort());
			return port;
		}
		
		return 0L;
	}
	else if(strcmp(name, "CVS_RCMD_PORT") == 0)
	{
		if(gCvsPrefs.RhostPort() != -1)
		{
			sprintf(port, "%d", gCvsPrefs.RhostPort());
			return port;
		}
		
		return 0L;
	}
	else if(strcmp(name, "CVS_SERVER") == 0)
	{
		return (char *)gCvsPrefs.ServerName();
	}
	else if(strcmp(name, "CVS_RSH") == 0)
	{
		if(gAuthen.kind() == ssh && gCvsPrefs.RshName() == 0L)
			return "ssh";
		return (char *)gCvsPrefs.RshName();
	}
	else if(strcmp(name, "CVSLIB_YESNO") == 0)
	{
		return (char *)AskYesNo();
	}
	else if(strcmp(name, "DIRTY_MCP") == 0)
	{
		return gCvsPrefs.DirtySupport() ? "yes" : "no";
	}
#ifdef WIN32
	else if(strcmp(name, "HOME") == 0)
	{
		// prompt for home the first time
		if(gCvsPrefs.Home() == 0L)
		{
			char oldpath[512];
			getcwd(oldpath, 511);
			const char *chome = BrowserGetDirectory("Select your home directory :");
			if(chome == 0L)
			{
				chdir(oldpath);
				return 0L;
			}

			gCvsPrefs.SetHome(chome);
			gCvsPrefs.save();
			chdir(oldpath);
		}

		// Windows 98 doesn't like C:\/.cvspass
		static CStr home;
		home = gCvsPrefs.Home();
		if(home.endsWith(kPathDelimiter))
		{
			home[home.length() - 1] = '\0';
		}

		return home;
	}
	else if(strcmp(name, "DLLLIBSTOP") == 0)
	{
		CWincvsApp* app = (CWincvsApp *)AfxGetApp();
		return app->gCvsStopping ? (char *)-1 : 0L;
	}
#endif /*WIN32 */
#ifdef macintosh
	else if(strcmp(name, "HOME") == 0)
	{
		FSSpec theFolder;
		static char thePath[260];
		Str255 aPath;
		
		if(MacGetPrefsFolder(theFolder, aPath) != noErr)
		{
			cvs_err("Unable to locate the preferences folder !\n");
			return 0L;
		}
		
		// We do not use any more the "@@@" trick. Instead
		// we let cvs convert it to a Unix path.
		//strcpy(thePath, "@@@");
		p2cstr(aPath);
		strcpy(thePath, (char *)aPath);
		return thePath;
	}
	else if(strcmp(name, "ISO8859") == 0)
	{
		static char iso[6];
		
		if(gCvsPrefs.IsoConvert() == ISO8559_none)
			return 0L;
		
		sprintf(iso, "%d", (int)gCvsPrefs.IsoConvert());
		return iso;
	}
	else if(strcmp(name, "IC_ON_TXT") == 0)
	{
		return gCvsPrefs.ICTextOption() ? "1" : 0L;
	}
	else if(strcmp(name, "MAC_DEFAULT_RESOURCE_ENCODING") == 0)
	{
		switch(gCvsPrefs.MacBinEncoding())
		{
			case MAC_HQX:
				return (char *)"HQX";
				break;
			case MAC_APPSINGLE:
				return (char *)"AppleSingle";
				break;
		}
		return 0L;
	}
#	ifdef qMacCvsPP
	else if(strcmp(name, "MAC_BINARY_TYPES_SINGLE") == 0)
	{
		return (char *)MacBinMaps::GetEnvTypes(MAC_APPSINGLE);
	}
	else if(strcmp(name, "MAC_BINARY_TYPES_HQX") == 0)
	{
		return (char *)MacBinMaps::GetEnvTypes(MAC_HQX);
	}
	else if(strcmp(name, "MAC_BINARY_TYPES_PLAIN") == 0)
	{
		return (char *)MacBinMaps::GetEnvPlainTypes();
	}
#	endif /* qMacCvsPP */
#endif /* macintosh */
#ifdef qQT
	else if(strcmp(name, "DLLLIBSTOP") == 0)
	{
		flushConsoles();
		QCvsApp::qApp->processEvents(50);
		return QCvsApp::qApp->IsCvsStopping() ? (char *)-1 : 0L;
	}
	else
		return getenv(name);
#endif /* qQT */
	
	return NULL;
}

static bool loadCVS(CompatConnectID & connID)
{
#ifdef WIN32
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	if(app->gCvsRunning)
	{
		cvs_err("Error : cvs is already running !\n");
		return false;
	}
#endif /* WIN32 */
#ifdef qQT
	if(!dupConsoles())
	{
		restoreConsoles();
		return false;
	}
#endif /* qQT */

	int result = 0;
	const char *name_dll2;
	
	// get cvs shared library
#ifdef macintosh
#	if qDebug
		name_dll2 = "cvs2DbgLib";
#	else
		name_dll2 = "cvs2Lib";
#	endif
#endif /* macintosh */
#ifdef WIN32
	name_dll2 = "cvs2ntlib.dll";
#endif /* WIN32 */
#ifdef qQT
	name_dll2 = "libcvsshl.so";
#endif /* qQT */

	// open the library
	if(!CompatLoadLibrary(&connID, name_dll2))
	{
		cvs_err("Error: Unable to load the library %s\n", name_dll2);
		return false;
	}

	// set the console out
	if(!CompatCallLibrary1(connID, result, uppSetGlueProcInfo, "dllglue_setconsoleout",
		dllglue_setglue_func, CompatFuncArg(uppConsoleOutProcInfo, consoleout)))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_setconsoleout""\n");
		goto closeandfail;
	}

	// set the console err
	if(!CompatCallLibrary1(connID, result, uppSetGlueProcInfo, "dllglue_setconsoleerr",
		dllglue_setglue_func, CompatFuncArg(uppConsoleOutProcInfo, consoleerr)))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_setconsoleerr""\n");
		goto closeandfail;
	}

	// set the console in
	if(!CompatCallLibrary1(connID, result, uppSetGlueProcInfo, "dllglue_setconsolein",
		dllglue_setglue_func, CompatFuncArg(uppConsoleInProcInfo, consolein)))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_setconsolein""\n");
		goto closeandfail;
	}

	// set the getenv
	if(!CompatCallLibrary1(connID, result, uppSetGlueProcInfo, "dllglue_setgetenv",
		dllglue_setglue_func, CompatFuncArg(uppGetEnvProcInfo, mygetenv)))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_setgetenv""\n");
		goto closeandfail;
	}

#ifdef macintosh
	// set the event dispatcher
	if(!CompatCallLibrary1(connID, result, uppSetGlueProcInfo, "dllglue_seteventdispatch",
		dllglue_setglue_func, CompatFuncArg(uppEventDispatchProcInfo, myeventdispatcher)))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_seteventdispatch""\n");
		goto closeandfail;
	}
#endif /* macintosh */

#ifdef qMacCvsPP
	CMacCvsApp::app->cvsLoaded = true;
#endif /* qMacCvsPP */
#ifdef qQT
	QCvsApp::qApp->SetCvsLoaded(true);
	QCvsApp::qApp->SetCvsStopping(false);
#endif /* qQT */
		
	return true;
closeandfail:
	CompatCloseLibrary(&connID);
	return false;
}

static void unloadCVS(CompatConnectID connID, int exitc)
{	
	CompatCloseLibrary(&connID);
	
	cvs_out("\n*****CVS exited normally with code %d*****\n\n", exitc);

#ifdef qMacCvsPP
	CMacCvsApp::app->cvsLoaded = false;
	CMacCvsApp::app->RefreshBrowsers();
#endif /* qMacCvsPP */
#ifdef WIN32
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	if(app->gCvsStopping)
	{
		cvs_out("*****CVS stopped on user request !!! *****\n\n");
	}
#endif /* WIN32 */
#ifdef qQT
	QCvsApp::qApp->SetCvsLoaded(false);
	if(QCvsApp::qApp->IsCvsStopping())
	{
		cvs_out("*****CVS stopped on user request !!! *****\n\n");
	}
	QCvsApp::qApp->SetCvsStopping(false);
	restoreConsoles();
	QCvsApp::qApp->GetFileView()->ResetView();
#endif /* qQT */
}

#ifdef WIN32
// critical section to protect while accessing MFC from the DLL
CRITICAL_SECTION CCvsThread::m_lock;

CCvsThread::CCvsThread()
{
}

void CCvsThread::Delete()
{
	// calling the base here won't do anything but it is a good habit
	CWinThread::Delete();
}

void CCvsThread::KillThread()
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	app->gCvsStopping = true;
}

CCvsThread::~CCvsThread()
{
	sLastWin32CvsThread = 0L;
}

BOOL CCvsThread::InitInstance()
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();

	if(!loadCVS(mConnID))
		return FALSE;
	
	app->gCvsRunning = true;
	app->gCvsStopping = false;

	app->gLastCvsResult = Launch();

	unloadCVS(mConnID, app->gLastCvsResult);

	app->gCvsRunning = false;
	app->gCvsStopping = false;

	app->GetFileView()->CheckChanges();

	return FALSE;
}

IMPLEMENT_DYNAMIC(CCvsThread, CWinThread);

BEGIN_MESSAGE_MAP(CCvsThread, CWinThread)
	//{{AFX_MSG_MAP(CCvsThread)
		// NOTE - the ClassWizard will add and remove mapping macros here.
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

class CCvsCmdThread : public CCvsThread
{
public:
	DECLARE_DYNAMIC(CCvsCmdThread)

	CCvsCmdThread(const char *path, int argc, char * const *argv);
	virtual ~CCvsCmdThread();

// Overrides
	// ClassWizard generated virtual function overrides
	//{{AFX_VIRTUAL(CCvsCmdThread)
	//}}AFX_VIRTUAL

protected:
	virtual int Launch();

	CvsArgs mArgs;
	CPStr mPath;

	// Generated message map functions
	//{{AFX_MSG(CCvsCmdThread)
		// NOTE - the ClassWizard will add and remove member functions here.
	//}}AFX_MSG

	DECLARE_MESSAGE_MAP()
};

CCvsCmdThread::CCvsCmdThread(const char *path, int argc, char * const *argv) :
		mArgs(argv, argc), mPath(path), CCvsThread()
{
}

CCvsCmdThread::~CCvsCmdThread()
{
}

int CCvsCmdThread::Launch()
{
	int exitc = 0;

	if(!CompatCallLibrary3(mConnID, exitc, uppMainProcInfo, "dllglue_main",
		dllglue_main_func, mPath.empty() ? 0L : (const char *)mPath,
		mArgs.Argc(), (char **)mArgs.Argv()))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_main""\n");
		exitc = 1;
	}
	return exitc;
}

IMPLEMENT_DYNAMIC(CCvsCmdThread, CCvsThread);

BEGIN_MESSAGE_MAP(CCvsCmdThread, CCvsThread)
	//{{AFX_MSG_MAP(CCvsCmdThread)
		// NOTE - the ClassWizard will add and remove mapping macros here.
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

#endif /* WIN32 */

int launchCVS(const char *path, int argc, char * const *argv, CCvsConsole * console)
{
	sConsole = console;
	
#ifdef WIN32
	// sorry for Windows 98 and Windows 95,
	// it takes too much time to launch the thread...
	if(/*gCvsPrefs.WhichVersion() == kWin32 || */console != 0L)
	{
	sLastWin32CvsThread = 0L;
#endif /* WIN32 */
	CompatConnectID connID;
	
	if(!loadCVS(connID))
		return errInternal;
	
	int exitc = 0;

	// call the main
	if(!CompatCallLibrary3(connID, exitc, uppMainProcInfo, "dllglue_main",
		dllglue_main_func, path, argc, (char **)argv))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_main""\n");
		exitc = errInternal;
	}

	unloadCVS(connID, exitc);
	
	return exitc;
#ifdef WIN32
	}
	else
	{
	CCvsCmdThread* pThread;
	pThread = new CCvsCmdThread(path, argc, (char **)argv);
	if (pThread == 0L)
	{
		cvs_err("Impossible to create the cvs thread !\n");
		return errInternal;
	}

	ASSERT_VALID(pThread);

	// Create Thread in a suspended state so we can set the Priority 
	// before it starts getting away from us
	if (!pThread->CreateThread(CREATE_SUSPENDED))
	{
		cvs_err("Impossible to start the cvs thread !\n");
		delete pThread;
		return errInternal;
	}

	// If you want to make the sample more sprightly, set the thread priority here 
	// a little higher. It has been set at idle priority to keep from bogging down 
	// other apps that may also be running.
	VERIFY(pThread->SetThreadPriority(THREAD_PRIORITY_NORMAL));
	// Now the thread can run wild
	pThread->ResumeThread();
	sLastWin32CvsThread = pThread;

	return errThreadRunning;
	}
#endif /* WIN32 */
}

class CStreamConsole : public CCvsConsole
{
public:
	CStreamConsole(FILE *out) : fOut(out) {}

	virtual long cvs_out(char *txt, long len)
	{
		fwrite(txt, sizeof(char), len, fOut);
		return len;
	}
	virtual long cvs_err(char *txt, long len)
	{
		cvs_errstr(txt, len);
		return len;
	}
protected:
	FILE *fOut;
};

FILE *launchCVS(const char *dir, const CvsArgs & args, CStr & tmpFile)
{
#ifdef WIN32
	tmpFile = _tempnam(0L, 0L);
#endif /* !WIN32 */
#ifdef macintosh
	CStr tmpname("_maccvsXXXXXX");
	CStr tmpPath;
	FSSpec theFolder;
	Str255 aPath;
	OSErr err;
	
	if((err = MacGetPrefsFolder(theFolder, aPath)) != noErr)
	{
		cvs_err("Unable to locate the preferences folder (error %d) !\n", err);
		return 0L;
	}
	tmpPath = aPath;
	if(chdir(tmpPath) != 0)
	{
		cvs_err("Unable to chdir to '%s' (error %d) !\n", (const char *)tmpPath, errno);
		return 0L;
	}
	if(!tmpPath.endsWith(kPathDelimiter))
		tmpPath << kPathDelimiter;
	tmpPath << mktemp(tmpname);
	if(chdir(dir) != 0)
	{
		cvs_err("Unable to chdir to '%s' (error %d) !\n", dir, errno);
		return 0L;
	}
	tmpFile = tmpPath;
#endif /* !macintosh */

	FILE * res = fopen(tmpFile, "w+");
	if(res == 0L)
	{
		cvs_err("Impossible to open for write the temp file '%s' (error %d)",
			(const char *)tmpFile, errno);
		return 0L;
	}

	CStreamConsole myconsole(res);

	launchCVS(dir, args.Argc(), args.Argv(), &myconsole);

	fseek(res, 0L, SEEK_SET);

	return res;
}

class CTmpFiles
{
public:
	~CTmpFiles()
	{
		vector<CStr>::const_iterator i;
		for(i = allfiles.begin(); i != allfiles.end(); ++i)
		{
			const char *file = *i;
			unlink((char *)file);
		}
	}

	inline const char *push_back(const char *file)
	{
		allfiles.push_back(file);
		return allfiles[allfiles.size() - 1];
	}
protected:
	vector<CStr> allfiles;
} sTmpList;

const char *launchCVS(const char *dir, const CvsArgs & args)
{
	CStr filename;
	FILE *file = launchCVS(dir, args, filename);
	if(file == 0L)
		return 0L;

	fclose(file);
	return sTmpList.push_back(filename);
}

#ifdef WIN32
void WaitForCvs(void)
{
	// pump messages manually
	while(1)
	{
		if(sLastWin32CvsThread == 0L)
			return;

		DWORD dw = ::WaitForSingleObject(*sLastWin32CvsThread, 300);

		switch(dw) 
		{ 
			case WAIT_OBJECT_0: 
			case WAIT_TIMEOUT: 
				break;
			case WAIT_ABANDONED: 
			default: 
				cvs_err("An error occured while waiting for cvs to return (error %d)\n", ::GetLastError());
				return;
				break;
		}

		// pump message, but quit on WM_QUIT
		if (!AfxGetThread()->PumpMessage())
		{
			AfxPostQuitMessage(0);
			return;
		}
	}
}
#endif
