/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- January 1997
 */

/*
 * CPStr.h --- simple Pascal-C string
 */

#ifndef CPSTR_H
#define CPSTR_H

#include <string.h>
#include <stdlib.h>

#include "Persistent.h"

// up to 255 characters
class CPStr
{
public :
	inline CPStr() : str(0L) {}
	inline CPStr(const char *newstr)
	{
		str = 0L;
		*this = newstr;
	}
	inline CPStr(const unsigned char *newstr)
	{
		str = 0L;
		*this = newstr;
	}
	inline CPStr(const CPStr & newstr)
	{
		str = 0L;
		*this = newstr;
	}
	virtual ~CPStr() {flush();}

	inline bool empty(void) const {return str == 0L || str[0] == '\0';}
	inline unsigned int length(void) const {return str == 0L ? 0 : (unsigned char)str[0];}

	const char *operator=(const char *newstr);
		// set to a new C String (0L is OK)
	const unsigned char *operator=(const unsigned char *newstr);
		// set to a new P String (0L is OK)
	const CPStr & operator=(const CPStr & newstr);
		// set according to another CPStr
	const CPStr & set(const char *buf, unsigned int len);
		// set from a buffer

	inline operator const char *() const
		{return str == 0L ? "" : str + 1;}
	inline operator char *()
		{return str == 0L ? "" : str + 1;}
		// as a C string
	inline operator const unsigned char *() const
		{return str == 0L ? (unsigned char *)"" : (unsigned char *)str;}
	inline operator unsigned char *()
		{return str == 0L ? (unsigned char *)"" : (unsigned char *)str;}
		// as a P string

	CPStr & operator<<(const char *addToStr);
	CPStr & operator<<(char addToStr);
	CPStr & operator<<(int addToStr);
		// concatenate

	inline bool endsWith(char c) const {return str == 0L ? false : str[str[0]] == c;}
	
	const CPStr & replace(char which, char bywhich);
		// replace a character
	
protected :
	void flush(void);
	char *str; // the String
};

// above 255 characters, no-Pascal string support
class CStr
{
public :
	inline CStr() : str(0L) {}
	inline CStr(const char *newstr)
	{
		str = 0L;
		*this = newstr;
	}
	inline CStr(const CStr & newstr)
	{
		str = 0L;
		*this = newstr;
	}
	virtual ~CStr() {flush();}

	// CVSROOT
	inline bool empty(void) const {return str == 0L || str[0] == '\0';}
	inline unsigned int length(void) const {return str == 0L ? 0 : strlen(str);}

	const char *operator=(const char *newstr);
		// set from a C String (0L is OK)
	const char *operator=(const unsigned char *newstr);
		// set from a P String (0L is OK)
	const CStr & operator=(const CStr & newstr);
		// set according to another CStr
	const CStr & set(const char *buf, unsigned int len);
		// set from a buffer

#ifdef WIN32
	bool operator<(const CStr & str) const;
		// not implemented : only provided to compile
		// with STL
#endif /* WIN32 */

#ifdef macintosh
	inline char operator[](int index) const { return str[index]; }
	inline char & operator[](int index) { return str[index]; }
#endif /* macintosh */

	inline operator const char *() const
		{return str == 0L ? "" : str;}
	inline operator char *()
		{return str == 0L ? "" : str;}
		// as a C string

	CStr & operator<<(const char *addToStr);
	CStr & operator<<(char addToStr);
	CStr & operator<<(int addToStr);
		// concatenate

	inline int compare(const char *thestr) const { return strcmp(*this, thestr); }
		// compare

	inline bool endsWith(char c) const {return str == 0L ? false : str[length()-1] == c;}
	
	const CStr & replace(char which, char bywhich);
		// replace a character
	
protected :
	void flush(void);
	char *str; // the String
};

// the persistent version
class PCPStr : public CPStr, public CPersistent
{
public:
	PCPStr(const char *uniqueName, const char *defValue = 0L) : CPStr()
	{
		if(defValue != 0L)
			*(CPStr *)this = defValue;
		CPersistent::Register(this, uniqueName);
	}
	virtual ~PCPStr()
	{
		CPersistent::UnRegister(this);
	}

	virtual unsigned int SizeOf(void) const
	{
		return str != 0L ? length() + 1 : 1;
	}
	
	virtual const void *GetData(void) const
	{
		return (const char *)(*(CPStr *)this);
	}
	
	virtual void SetData(const void *ptr, unsigned int size)
	{
		if(strlen((const char *)ptr) == (size - 1))
			*(CPStr *)this = (const char *)ptr;
	}
};

class PCStr : public CStr, public CPersistent
{
public:
	PCStr(const char *uniqueName, const char *defValue = 0L) : CStr()
	{
		if(defValue != 0L)
			*(CStr *)this = defValue;
		CPersistent::Register(this, uniqueName);
	}
	virtual ~PCStr()
	{
		CPersistent::UnRegister(this);
	}

	virtual unsigned int SizeOf(void) const
	{
		return str != 0L ? length() + 1 : 0;
	}
	
	virtual const void *GetData(void) const
	{
		return (const char *)(*(CStr *)this);
	}
	
	virtual void SetData(const void *ptr, unsigned int size)
	{
		if(strlen((const char *)ptr) == (size - 1))
			*(CStr *)this = (const char *)ptr;
	}
};

// class to use when fed-up with realloc, malloc...
template <class T>
class CStaticAllocT
{
public:
	CStaticAllocT()
	{
		buf = 0L;
		numbuf = 0;
	}

	~CStaticAllocT()
	{
		if(buf != 0L)
			free(buf);
	}

	void AdjustSize(size_t size)
	{
		if(numbuf < size)
		{
			if(buf == 0L)
				buf = (T *)malloc(size * sizeof(T));
			else
				buf = (T *)realloc(buf, size * sizeof(T));
			numbuf = size;
		}
	}

	inline operator T *(void) { return buf; }
	inline operator const T *(void) const { return buf; }

	inline size_t size() const { return numbuf; }
protected:
	T *buf;
	size_t numbuf;
};

// Modena STL support
#ifdef __MWERKS__
	SPECIALIZE_ITERATOR_TRAIT(CStr)
	SPECIALIZE_ITERATOR_TRAIT(CPStr)
#endif /* __MWERKS__ */

#endif /* CPSTR_H */
