/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- April 1998
 */

/*
 * CvsEntries.cpp --- adaptation from cvs/src/entries.c
 */

#include "stdafx.h"

#include <stdlib.h>
#include <string.h>

#ifdef qMacCvsPP
#	include <GUSI.h>
#	include <time.h>
#	include <stat.h>
#else /* !qMacCvsPP */
#	include <sys/stat.h>
#endif /* !qMacCvsPP */

#include "CvsEntries.h"
#include "AppConsole.h"
#include "getline.h"

#ifdef WIN32
#	ifdef _DEBUG
#		define new DEBUG_NEW
#		undef THIS_FILE
		static char THIS_FILE[] = __FILE__;
#	endif
#endif /* WIN32 */

#if TIME_WITH_SYS_TIME
#	include <sys/time.h>
#	include <time.h>
#elif HAVE_SYS_TIME_H
#	include <sys/time.h>
#else
#	include <time.h>
#endif

#ifdef macintosh
static char *strdup(const char *string)
{
	int size = strlen(string);
	char *result = (char *)malloc((size + 1) * sizeof(char));
	if(result == 0L)
		return 0L;
	strcpy(result, string);
	return result;
}
#endif /* macintosh */

EntnodeData::~EntnodeData()
{
	--ref;
	if(user != 0L)
		free(user);
	if(desc != 0L)
		free(desc);
}

void EntnodeData::SetDesc(const char *newdesc)
{
	if(desc != 0L)
	{
		free(desc);
		desc = 0L;
	}
	if(newdesc != 0L && newdesc[0] != '\0' && (desc = strdup(newdesc)) == 0L)
		throw ENOMEM;
}

EntnodeFile::~EntnodeFile()
{
	if(vn != 0L)
		free(vn);
	if(ts != 0L)
		free(ts);
	if(option != 0L)
		free(option);
	if(tag != 0L)
		free(tag);
	if(date != 0L)
		free(date);
	if(ts_conflict != 0L)
		free(ts_conflict);
}

EntnodeFile::EntnodeFile(const char *newuser, const char *newvn,
	const char *newts, const char *newoptions, const char *newtag,
	const char *newdate, const char *newts_conflict) : EntnodeData(ENT_FILE)
{
	vn = 0L;
	ts = 0L;
	option = 0L;
	tag = 0L;
	date = 0L;
	ts_conflict = 0L;

	if(newuser != 0L && newuser[0] != '\0' && (user = strdup(newuser)) == 0L)
		throw ENOMEM;
	if(newvn != 0L && newvn[0] != '\0' && (vn = strdup(newvn)) == 0L)
		throw ENOMEM;
	if(newts != 0L && newts[0] != '\0' && (ts = strdup(newts)) == 0L)
		throw ENOMEM;
	if(newoptions != 0L && newoptions[0] != '\0' && (option = strdup(newoptions)) == 0L)
		throw ENOMEM;
	if(newtag != 0L && newtag[0] != '\0' && (tag = strdup(newtag)) == 0L)
		throw ENOMEM;
	if(newdate != 0L && newdate[0] != '\0' && (date = strdup(newdate)) == 0L)
		throw ENOMEM;
	if(newts_conflict != 0L && newts_conflict[0] != '\0' && (ts_conflict = strdup(newts_conflict)) == 0L)
		throw ENOMEM;
}

EntnodeFolder::~EntnodeFolder()
{
}

EntnodeFolder::EntnodeFolder(const char *newuser, const char *newvn,
	const char *newts, const char *newoptions, const char *newtag,
	const char *newdate, const char *newts_conflict) : EntnodeData(ENT_SUBDIR)
{
	if(newuser != 0L && newuser[0] != '\0' && (user = strdup(newuser)) == 0L)
		throw ENOMEM;
}

/* Return the next real Entries line.  On end of file, returns NULL.
   On error, prints an error message and returns NULL.  */

static EntnodeData * fgetentent(FILE *fpin, char *cmd = 0L)
{
	EntnodeData *ent;
	char *line = 0L;
	size_t line_chars_allocated = 0;
	register char *cp;
	ent_type type;
	char *l, *user, *vn, *ts, *options;
	char *tag_or_date, *tag, *date, *ts_conflict;
	int line_length;

	ent = 0L;
	while ((line_length = getline (&line, &line_chars_allocated, fpin)) > 0)
	{
		l = line;

		/* If CMD is not NULL, we are reading an Entries.Log file.
		Each line in the Entries.Log file starts with a single
		character command followed by a space.  For backward
		compatibility, the absence of a space indicates an add
		command.  */
		if (cmd != 0L)
		{
			if (l[1] != ' ')
				*cmd = 'A';
			else
			{
				*cmd = l[0];
				l += 2;
			}
		}
		type = ENT_FILE;

		if (l[0] == 'D')
		{
			type = ENT_SUBDIR;
			++l;
			/* An empty D line is permitted; it is a signal that this
			Entries file lists all known subdirectories.  */
		}

		if (l[0] != '/')
			continue;

		user = l + 1;
		if ((cp = strchr (user, '/')) == 0L)
			continue;
		*cp++ = '\0';
		vn = cp;
		if ((cp = strchr (vn, '/')) == 0L)
			continue;
		*cp++ = '\0';
		ts = cp;
		if ((cp = strchr (ts, '/')) == 0L)
			continue;
		*cp++ = '\0';
		options = cp;
		if ((cp = strchr (options, '/')) == 0L)
			continue;
		*cp++ = '\0';
		tag_or_date = cp;
		if ((cp = strchr (tag_or_date, '\n')) == 0L)
			continue;
		*cp = '\0';
		tag = (char *) 0L;
		date = (char *) 0L;
		if (*tag_or_date == 'T')
			tag = tag_or_date + 1;
		else if (*tag_or_date == 'D')
			date = tag_or_date + 1;

		if ((ts_conflict = strchr (ts, '+')))
			*ts_conflict++ = '\0';

		if(type == ENT_SUBDIR)
			ent = new EntnodeFolder(user);
		else
			ent = new EntnodeFile(user, vn, ts, options,
				tag, date, ts_conflict);
		if(ent == 0L)
		{
			errno = ENOMEM;
			cvs_err("Out of memory\n");
		}
		break;
	}

	if (line_length < 0 && !feof (fpin))
		cvs_err("Cannot read entries file (error %d)\n", errno);

	free (line);
	return ent;
}

/* Read the entries file into a list, hashing on the file name.

   UPDATE_DIR is the name of the current directory, for use in error
   messages, or NULL if not known (that is, noone has gotten around
   to updating the caller to pass in the information).  */
bool Entries_Open (CSortList<ENTNODE> & entries, const char *fullpath)
{
	EntnodeData *ent;

	entries.Reset();
	FILE *fpin = fopen("Entries", "r");
	if (fpin == 0L)
		return false;

	while ((ent = fgetentent (fpin)) != 0L)
	{
		ENTNODE newnode(ent);
		ent->UnRef();

		if(entries.InsideAndInsert(newnode))
		{
			cvs_err("Warning : duplicated entry in the 'CVS/Entries' file in directory '%s'\n",
				fullpath);
		}
	}

	fclose (fpin);

	fpin = fopen("Entries.log", "r");
	if (fpin != 0L)
	{
		char cmd;

		while ((ent = fgetentent (fpin, &cmd)) != 0L)
		{
			ENTNODE newnode(ent);
			ent->UnRef();

			switch (cmd)
			{
			case 'A':
				if(entries.InsideAndInsert(newnode))
				{
					// dont care
				}
				break;
			case 'R':
				entries.Delete(newnode);
				break;
			default:
				/* Ignore unrecognized commands.  */
				break;
			}
		}
		fclose (fpin);
	}
	return true;
}

// mostly the cvs one...
static bool unmodified(const struct stat & sb, const char *ts)
{
    char *cp;
	struct tm *tm_p;
	struct tm local_tm;
	/* We want to use the same timestamp format as is stored in the
	st_mtime.  For unix (and NT I think) this *must* be universal
	time (UT), so that files don't appear to be modified merely
	because the timezone has changed.  For VMS, or hopefully other
	systems where gmtime returns NULL, the modification time is
	stored in local time, and therefore it is not possible to cause
	st_mtime to be out of sync by changing the timezone.  */
	tm_p = gmtime (&sb.st_mtime);
	if (tm_p)
	{
		memcpy (&local_tm, tm_p, sizeof (local_tm));
		cp = asctime (&local_tm);	/* copy in the modify time */
	}
	else
		cp = ctime(&sb.st_mtime);

	cp[24] = 0;

	return strcmp(cp, ts) == 0;
}

EntnodeData *Entries_SetVisited(CSortList<ENTNODE> & entries, const char *name,
	const struct stat & finfo, bool isFolder, const vector<CStr> * ignlist)
{
	int index;
	bool isCvs;
	if(isFolder)
	{
		EntnodeFolder *adata = new EntnodeFolder(name);
		ENTNODE anode(adata);
		adata->UnRef();
		isCvs = entries.InsideAndInsert(anode, &index);
	}
	else
	{
		EntnodeFile *adata = new EntnodeFile(name);
		ENTNODE anode(adata);
		adata->UnRef();
		isCvs = entries.InsideAndInsert(anode, &index);
	}

	const ENTNODE & theNode = entries.Get(index);
	EntnodeData *data = ((ENTNODE *)&theNode)->Data();
	data->SetVisited(true);
	if(!isCvs)
	{
		data->SetUnknown(true);
		if(ignlist != 0L && MatchIgnoredList(name, *ignlist))
			data->SetIgnored(true);
	}

	if(isFolder)
	{
		if(data->IsIgnored())
		{
			data->SetDesc("Ignored Folder");
		}
		else if(data->IsUnknown())
		{
			data->SetDesc("Unknown Folder");
		}
		else
		{
			data->SetDesc("Folder");
		}
	}
	else
	{
		const char *ts = (*data)[EntnodeFile::kTS];
		if(ts == 0L)
			data->SetUnmodified(true);
		else
			data->SetUnmodified(unmodified(finfo, ts));

		data->SetLocked((finfo.st_mode & S_IWRITE) == 0);
		
		const char *info = 0L;
		if(data->IsIgnored())
		{
			data->SetDesc("Ignored");
		}
		else if(data->IsUnknown())
		{
			data->SetDesc("Unknown");
		}
		else if((*data)[EntnodeFile::kConflict] != 0L)
		{
			data->SetDesc("Conflict");
		}
		else if((info = (*data)[EntnodeFile::kOption]) != 0L && strcmp(info, "-kb") == 0)
		{
			data->SetDesc(data->IsUnmodified() ? "Binary" : "Mod. Binary");
		}
		else
		{
			data->SetDesc(data->IsUnmodified() ? "File" : "Mod. File");
		}
	}

	return data;
}

void Entries_SetMissing(CSortList<ENTNODE> & entries)
{
	int numEntries = entries.NumOfElements();
	for(int i = 0; i < numEntries; i++)
	{
		const ENTNODE & theNode = entries.Get(i);
		EntnodeData *data = ((ENTNODE *)&theNode)->Data();
		if(data->IsVisited())
			continue;

		data->SetMissing(true);
		data->SetDesc("Missing");
	}
}

bool Tag_Open(CStr & tag, const char *fullpath)
{
	tag = "";

	FILE *fpin = fopen("Tag", "r");
	if (fpin == 0L)
		return false;

	char *line = 0L;
	size_t line_chars_allocated = 0;
	int line_length;
	if((line_length = getline (&line, &line_chars_allocated, fpin)) > 0)
	{
		char *tmp = strchr(line, '\n');
		if(tmp != 0L)
			*tmp = '\0';
		if(line[0] == 'T')
			tag = line + 1;
	}
	if(line != 0L)
		free(line);
	if (line_length < 0 && !feof (fpin))
	{
		cvs_err("Cannot open Tag file in '%s' (error %d)\n", fullpath, errno);
		return false;
	}

	fclose (fpin);
	return !tag.empty();
}
