/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- April 1998
 */

/*
 * FileTraversal.cpp : class to traverse a file hierarchy
 */

#include "stdafx.h"

#ifdef WIN32
#	include "resource.h"
#endif /* WIN32 */

#ifdef macintosh
#	include <GUSI.h>
#	include <TFileSpec.h>
#	include <errno.h>
#endif /* macintosh */

#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif /* HAVE_UNISTD_H */

#ifdef HAVE_ERRNO_H
#	include <errno.h>
#endif /* HAVE_ERRNO_H */

#include "FileTraversal.h"
#include "CPStr.h"

#ifdef WIN32
#	ifdef _DEBUG
#	define new DEBUG_NEW
#	undef THIS_FILE
	static char THIS_FILE[] = __FILE__;
#	endif
#endif /* WIN32 */

kTraversal FileTraverse(const char *path, TraversalReport & report)
{
	DIR *dirp;
	STRUCT_DIRENT *dp;
	struct stat sb;
	CStr fullname, errstr, dirname, tmpname;
	kTraversal res = kContinueTraversal;

	if(path == 0L || chdir(path) != 0 || (dirp = opendir(path)) == 0L)
	{
		errstr = "Error while accessing ";
		errstr << path;
		errstr << " (error " << errno << ')';
		return report.OnError(errstr, errno);
	}

	tmpname = path;
	char *tmp = tmpname;
	char *next;
	if(tmpname.endsWith(kPathDelimiter))
		tmp[strlen(tmp) - 1] = '\0';
	next = tmp;
	while((tmp = strchr(next, kPathDelimiter)) != 0L)
	{
		next = ++tmp;
	}
	dirname = next;

	res = report.EnterDirectory(path, dirname);
	if(res != kContinueTraversal)
		goto abort;

	while ((dp = readdir (dirp)) != NULL)
	{
		res = report.OnIdle(path);
		if(res != kContinueTraversal)
			goto abort;

#ifndef macintosh
		if(strcmp(dp->d_name, ".") == 0 || strcmp(dp->d_name, "..") == 0)
			continue;
#endif /* !macintosh */

		fullname = path;
		if(!fullname.endsWith(kPathDelimiter))
			fullname << kPathDelimiter;
		fullname << dp->d_name;

		FSSpec *macspec = 0L;
#ifdef macintosh
		TFileSpec inspec((const char *)fullname);
		if(TFileSpec::Error() != noErr)
		{
			errstr = "Error while accessing ";
			errstr << fullname;
			errstr << " (error " << TFileSpec::Error() << ')';
			res = report.OnError(errstr, errno);
			goto abort;
		}
		macspec = &inspec;
#endif /* macintosh */

#ifndef macintosh
		if (stat(dp->d_name, &sb) != -1)
#else /* macintosh */
		if (lstat(dp->d_name, &sb) != -1)
#endif /* macintosh */
		{
			if(S_ISDIR(sb.st_mode))
			{
				res = report.OnDirectory(path, fullname, dp->d_name, sb, macspec);

				if(chdir(path) != 0)
				{
					errstr = "Error while accessing ";
					errstr << path;
					errstr << " (error " << errno << ')';
					res = report.OnError(errstr, errno);
					goto abort;
				}

				if(res == kSkipFile)
					continue;
				if(res != kContinueTraversal)
					goto abort;

				res = FileTraverse(fullname, report);
				if(res != kContinueTraversal)
					goto abort;

				if(chdir(path) != 0)
				{
					errstr = "Error while accessing ";
					errstr << path;
					errstr << " (error " << errno << ')';
					res = report.OnError(errstr, errno);
					goto abort;
				}
			}
			else if(S_ISREG(sb.st_mode))
			{				
				res = report.OnFile(path, fullname, dp->d_name, sb, macspec);
				if(res == kSkipFile)
					continue;
				if(res != kContinueTraversal)
					goto abort;
			}
#ifdef S_ISLNK
			else if(S_ISLNK(sb.st_mode))
			{
				res = report.OnAlias(path, fullname, dp->d_name, sb, macspec);
				if(res == kSkipFile)
					continue;
				if(res != kContinueTraversal)
					goto abort;
			}
#endif /* S_ISLNK */
			else
			{
				/* TODO : Unix devices */
				errstr = "Error while accessing ";
				errstr << fullname;
				errstr << " (error " << errno << ')';
				res = report.OnError(errstr, errno);
				goto abort;
			}
		}
		else
		{
			errstr = "Error while accessing ";
			errstr << fullname;
			errstr << " (error " << errno << ')';
			res = report.OnError(errstr, errno);
			goto abort;
		}
	}
	
	res = report.ExitDirectory(path);

abort:
	closedir (dirp);
	return res;
}
