/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@hotmail.com> --- December 1997
 */

/*
 * AppGlue.cpp --- glue code to access CVS shared library
 */

#include "stdafx.h"

#ifdef macintosh
#	include <TextUtils.h>
#	include <Strings.h>
#	include <Dialogs.h>
#	include <unistd.h>
#	include <errno.h>
#	include <console.h>
#	include <ctype.h>
#	include "mktemp.h"
#endif /* macintosh */

#ifdef qMacCvsPP
#	include "MacCvsApp.h"
#	include "LogWindow.h"
#	include "MacBinEncoding.h"
#	include "MacCvsAE.h"
#endif /* qMacCvsPP */

#ifdef WIN32
#	include "wincvs.h"
#	include "wincvsView.h"
#	include "PromptFiles.h"
#	include "BrowseFileView.h"
#	include <process.h>
#	include <direct.h>

#	ifdef _DEBUG
#	define new DEBUG_NEW
#	undef THIS_FILE
	static char THIS_FILE[] = __FILE__;
#	endif
#endif /* WIN32 */

#ifdef qUnix
#	include "UCvsApp.h"
#	include "UCvsFiles.h"
#	include "cvsgui_process.h"
#	include <ctype.h>
#else /* !qUnix */
#	include "dll_loader.h"
#	include "dll_process.h"
#endif /* !qUnix */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#ifdef HAVE_ERRNO_H
#	include <errno.h>
#endif

#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif

#include <vector>

#include "Authen.h"
#include "CvsPrefs.h"
#include "AppGlue.h"
#include "AppConsole.h"
#include "GetPassword.h"
#include "AskYesNo.h"
#include "CvsArgs.h"
#include "FileTraversal.h"

// overide console
static CCvsConsole * sConsole = 0L;

#ifdef WIN32
	static CWinThread *sLastWin32CvsThread = 0L;
#endif

extern "C" {
	static long consoleout(char *txt, long len);
	static long consoleerr(char *txt, long len);
	static long consolein(char *txt, long len);
	static char *mygetenv(char *name);
#ifdef macintosh
	static int myeventdispatcher(EventRecord *ev);
#endif /* macintosh */
#if qUnix
	static void myexit(int code);
#endif
}

#if qUnix
static CvsProcessCallbacks sCallTable = {
	consoleout,
	consoleerr,
	mygetenv,
	myexit
};

static void myexit(int code)
{
	if(!gCvsPrefs.IsTclFileRunning())
		cvs_out("\n*****CVS exited normally with code %d*****\n\n", code);
}
#endif

#ifdef _MSC_VER
extern "C"
#endif
static long consoleout(char *txt, long len)
{
	if(sConsole != 0L)
		return sConsole->cvs_out(txt, len);
	
#ifdef macintosh
	// apple event output
	long newlen;
	if((newlen = ae_consoleout(txt, len)) != -1)
		return newlen;
	
#	ifdef qMacCvsPP
	CLogWindow *log = CMacCvsApp::app->GetLogWindow();
	if(log != 0L)
		log->WriteCharsToConsole(txt, len);
	return len;
#	else /* !qMacCvsPP */
	return WriteCharsToConsole (txt, len);
#	endif /* !qMacCvsPP */
#endif /* macintosh */
#ifdef WIN32
	CWinApp* app = AfxGetApp();
	//ASSERT(app->IsKindOf(RUNTIME_CLASS(CWincvsApp)));
	CWincvsView *view = ((CWincvsApp *)app)->GetConsoleView();
	if(view == NULL)
		return 0;

	view->OutConsole(txt, len);

	return len;
#endif /* WIN32 */
#ifdef qUnix
	cvs_outstr(txt, len);

	return len;
#endif /* qUnix */
}

#ifdef _MSC_VER
extern "C"
#endif
static long consoleerr(char *txt, long len)
{
	if(sConsole != 0L)
		return sConsole->cvs_err(txt, len);
	
#ifdef macintosh
	// apple event output
	long newlen;
	if((newlen = ae_consoleout(txt, len)) != -1)
		return newlen;

#	ifdef qMacCvsPP
	CLogWindow *log = CMacCvsApp::app->GetLogWindow();
	if(log != 0L)
		log->WriteCharsToConsole(txt, len, true);
	return len;
#	else /* !qMacCvsPP */
	return WriteCharsToConsole (txt, len);
#	endif /* !qMacCvsPP */
#endif /* macintosh */
#ifdef WIN32
	CWinApp* app = AfxGetApp();
	//ASSERT(app->IsKindOf(RUNTIME_CLASS(CWincvsApp)));
	CWincvsView *view = ((CWincvsApp *)app)->GetConsoleView();
	if(view == NULL)
		return 0;

	view->OutConsole(txt, len, true);

	return len;
#endif /* WIN32 */
#ifdef qUnix
	cvs_errstr(txt, len);

	return len;
#endif /* qUnix */
}

#ifdef _MSC_VER
extern "C"
#endif
static long consolein(char *txt, long len)
{
#ifdef macintosh
	// apple event input
	long newlen;
	if((newlen = ae_consolein(txt, len)) != -1)
		return newlen;

	return 0;
#else /* !macintosh */
	return 0;
#endif /* !macintosh */
}

#ifdef macintosh
extern "C" static int myeventdispatcher(EventRecord *ev)
{
#	ifdef qMacCvsPP
	EventRecord	localEvent;
	if (!ev) {
		//	Spin callback
		const	UInt32	kSpinSleep = 6;
		static	UInt32	spinTicks = 0;
		if ((::TickCount () - spinTicks) < kSpinSleep) return 0;
		
		ev = &localEvent;
		::WaitNextEvent (everyEvent, ev, kSpinSleep, nil);
		spinTicks = ::TickCount ();
		} // if
	
	if(ev->what == mouseDown)
	{
		WindowPtr macWindowP;
		SInt16 thePart = ::FindWindow(ev->where, &macWindowP);
		switch (thePart)
		{
			case inDrag:
			case inGrow:
			case inZoomIn:
			case inZoomOut:
				CMacCvsApp::app->DispatchEvent(*ev);
				break;
			case inContent:
				if (macWindowP != ::FrontWindow())
				{
					// Clicked Window is not the front one
					LWindow *frontW = LWindow::FetchWindowObject(macWindowP);
					if (frontW != nil && !UDesktop::WindowIsSelected(frontW))
						CMacCvsApp::app->DispatchEvent(*ev);
				}
				break;
		}
		return 0;
	}
	else if(ev->what == nullEvent)
	{
		CMacCvsApp::app->DoSpinCursor();
	}
	else if(ev->what != keyDown && ev->what != keyUp && ev->what != mouseUp)
		CMacCvsApp::app->DispatchEvent(*ev);
#	endif /* qMacCvsPP */
	return 0;
}
#endif /* macintosh */

#ifdef macintosh
#	define isGlobalEnvVariable(name) ( \
		strcmp(name, "CVS_PSERVER_PORT") == 0 || \
		strcmp(name, "CVS_RCMD_PORT") == 0 || \
		strcmp(name, "CVS_SERVER") == 0 || \
		strcmp(name, "CVS_RSH") == 0 || \
		strcmp(name, "CVS_RSA_IDENTITY") == 0 || \
		strcmp(name, "HOME") == 0 || \
		strcmp(name, "ISO8859") == 0 || \
		strcmp(name, "IC_ON_TXT") == 0 || \
		strcmp(name, "MAC_DEFAULT_RESOURCE_ENCODING") == 0 || \
		strcmp(name, "MAC_BINARY_TYPES_SINGLE") == 0 || \
		strcmp(name, "MAC_BINARY_TYPES_HQX") == 0 || \
		strcmp(name, "MAC_BINARY_TYPES_PLAIN") == 0 || \
		strcmp(name, "DIRTY_MCP") == 0 \
		)
#endif /* macintosh */

#ifdef _MSC_VER
extern "C"
#endif
static char *mygetenv(char *name)
{
#ifdef WIN32
	// synchonize it since the thread calls and
	// we call some MFC objects in it (prompt
	// password, get path...)
	CCvsThreadLock threadLock;
#endif /* WIN32 */

	static char port[10];

	if(name == NULL)
		return NULL;
		
#ifdef macintosh
	char *result;
	if((result = ae_getenv(name)) != (char *)-1)
	{
		if(result == 0L && isGlobalEnvVariable(name))
		{
			// in this case only we let continue
			// to get the MacCVS settings. So the CWCVS
			// users (or any AppleEvent user) can set
			// these variables in MacCVS but still can overide
			// them using Apple Events.
		}
		else
			return result;
	}
#endif /* macintosh */
	
	if(strcmp(name, "USER") == 0
#ifdef WIN32
		// I do that (and see the "STRATA_HACK" in
		// subr.c) because people get tired on Windows
		// to have to check their Windows login :
		// this is only for .rhosts authentication.
		|| strcmp(name, "LOGNAME") == 0
#endif /* WIN32 */
		)
	{
		static char user[64];
		
		// extract from the cvsroot
		const char *ccvsroot = gCvsPrefs;
		ccvsroot = Authen::skiptoken(ccvsroot);
		char *login = strchr(ccvsroot, '@');
		if(login == NULL)
		{
			// for WIN32 this means the CVSROOT is local
#ifdef macintosh
			cvs_err("MacCvs: Cannot extract login from cvsroot %s !\n", ccvsroot);
#endif /* macintosh */
			return NULL;
		}
		int length = login - ccvsroot;
		memcpy(user, ccvsroot, length * sizeof(char));
		user[length] = '\0';
		return user;
	}
	else if(strcmp(name, "CVSROOT") == 0)
	{
		static CStr root;
		
		// extract from the cvsroot
		const char *ccvsroot = gCvsPrefs;
		ccvsroot = Authen::skiptoken(ccvsroot);
		root = gAuthen.token();
		root << ccvsroot;
		return root;
	}
	else if(strcmp(name, "CVSREAD") == 0)
	{
		return gCvsPrefs.CheckoutRO() ? (char *)"1" : 0L;
	}
	else if(strcmp(name, "CVS_GETPASS") == 0)
	{
		return CompatGetPassword();
	}
	else if(strcmp(name, "CVS_PSERVER_PORT") == 0)
	{
		if(gCvsPrefs.PserverPort() != -1)
		{
			sprintf(port, "%d", gCvsPrefs.PserverPort());
			return port;
		}
		
		return 0L;
	}
	else if(strcmp(name, "CVS_RCMD_PORT") == 0)
	{
		if(gCvsPrefs.RhostPort() != -1)
		{
			sprintf(port, "%d", gCvsPrefs.RhostPort());
			return port;
		}
		
		return 0L;
	}
	else if(strcmp(name, "CVS_SERVER") == 0)
	{
		return (char *)gCvsPrefs.ServerName();
	}
	else if(strcmp(name, "CVS_RSH") == 0)
	{
		if(gAuthen.kind() == ssh && gCvsPrefs.RshName() == 0L)
			return "ssh";
		return (char *)gCvsPrefs.RshName();
	}
	else if( strcmp(name, "CVS_RSA_IDENTITY") == 0 )
	{
		if( ( gAuthen.kind() != ssh ) || !gAuthen.RsaIdentity() )
			return 0L;

		const char *tmp = gAuthen.RsaIdentityFile();
		if( tmp && *tmp )
		{
#ifdef WIN32
			while( *tmp && iswspace( *tmp ) )
#else
			while( *tmp && isspace( *tmp ) )
#endif
				tmp++;

			return *tmp ? (char *)tmp : 0L;
		}
		return 0L;
	}
	else if(strcmp(name, "CVSLIB_YESNO") == 0)
	{
		return (char *)AskYesNo();
	}
	else if(strcmp(name, "DIRTY_MCP") == 0)
	{
		return gCvsPrefs.DirtySupport() ? (char *)"yes" : (char *)"no";
	}
#ifdef WIN32
	else if(strcmp(name, "HOME") == 0)
	{
		// prompt for home the first time
		if(gCvsPrefs.Home() == 0L)
		{
			char oldpath[512];
			getcwd(oldpath, 511);
			const char *chome = BrowserGetDirectory("Select your home directory :");
			if(chome == 0L)
			{
				chdir(oldpath);
				return 0L;
			}

			gCvsPrefs.SetHome(chome);
			gCvsPrefs.save();
			chdir(oldpath);
		}

		// Windows 98 doesn't like C:\/.cvspass
		static CStr home;
		home = gCvsPrefs.Home();
		if(home.endsWith(kPathDelimiter))
		{
			home[home.length() - 1] = '\0';
		}

		return home;
	}
	else if(strcmp(name, "DLLLIBSTOP") == 0)
	{
		CWincvsApp* app = (CWincvsApp *)AfxGetApp();
		DWORD whatTimeIsIt = ::GetTickCount();
		static DWORD lastTime = 0;
		// let's make some stuff every second while cvs is idling
		if((whatTimeIsIt - lastTime) > 1000)
		{
			lastTime = whatTimeIsIt;

			// try to deal the WM_PAINT messages
			if(sLastWin32CvsThread == 0L)
			{
				MSG msg;
				while (PeekMessage(&msg, NULL, WM_PAINT, WM_PAINT, PM_NOREMOVE) == TRUE)
					app->PumpMessage();
			}
		}

		return app->gCvsStopping ? (char *)-1 : 0L;
	}
	else if(strcmp(name, "CVSUSEUNIXLF") == 0)
	{
		return gCvsPrefs.UnixLF() ? (char *)-1 : 0L;
	}
#endif /*WIN32 */
#ifdef macintosh
	else if(strcmp(name, "HOME") == 0)
	{
		FSSpec theFolder;
		static char thePath[256];
		Str255 aPath;
		
		if(MacGetPrefsFolder(theFolder, aPath) != noErr)
		{
			cvs_err("Unable to locate the preferences folder !\n");
			return 0L;
		}
		
		// We do not use any more the "@@@" trick. Instead
		// we let cvs convert it to a Unix path.
		//strcpy(thePath, "@@@");
		p2cstrcpy(thePath, aPath);
		return thePath;
	}
	else if(strcmp(name, "ISO8859") == 0)
	{
		static char iso[6];
		
		if(gCvsPrefs.IsoConvert() == ISO8559_none)
			return 0L;
		
		sprintf(iso, "%d", (int)gCvsPrefs.IsoConvert());
		return iso;
	}
	else if(strcmp(name, "IC_ON_TXT") == 0)
	{
		return gCvsPrefs.ICTextOption() ? (char *)"1" : 0L;
	}
	else if(strcmp(name, "MAC_DEFAULT_RESOURCE_ENCODING") == 0)
	{
		switch(gCvsPrefs.MacBinEncoding())
		{
			case MAC_HQX:
				return (char *)"HQX";
				break;
			case MAC_APPSINGLE:
				return (char *)"AppleSingle";
				break;
		}
		return 0L;
	}
#	ifdef qMacCvsPP
	else if(strcmp(name, "MAC_BINARY_TYPES_SINGLE") == 0)
	{
		return (char *)MacBinMaps::GetEnvTypes(MAC_APPSINGLE);
	}
	else if(strcmp(name, "MAC_BINARY_TYPES_HQX") == 0)
	{
		return (char *)MacBinMaps::GetEnvTypes(MAC_HQX);
	}
	else if(strcmp(name, "MAC_BINARY_TYPES_PLAIN") == 0)
	{
		return (char *)MacBinMaps::GetEnvPlainTypes();
	}
#	endif /* qMacCvsPP */
#endif /* macintosh */
#ifdef qUnix
	else
		return getenv(name);
#endif /* qUnix */
	
	return NULL;
}

#ifndef qUnix
static bool loadCVS(CompatConnectID & connID)
{
#ifdef WIN32
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	if(app->gCvsRunning)
	{
		cvs_err("Error : cvs is already running !\n");
		return false;
	}
#endif /* WIN32 */

	int result = 0;
	const char *name_dll2;
	
	// get cvs shared library
#ifdef macintosh
#	if qCarbon
#		if qCvsDebug
			name_dll2 = "cvsXDbgLib";
#		else
			name_dll2 = "cvsXLib";
#		endif
#	else
#		if qCvsDebug
			name_dll2 = "cvs2DbgLib";
#		else
			name_dll2 = "cvs2Lib";
#		endif
#	endif
#endif /* macintosh */
#ifdef WIN32
		if(gCvsPrefs.CvsVersion() == 0)
			name_dll2 = "cvs2ntlib.dll";
		else if(gCvsPrefs.WhichVersion() == kWin32)
			name_dll2 = "cvs2w95lib.dll";
		else
			name_dll2 = "cvs2ntslib.dll";
#endif /* WIN32 */

	// open the library
	if(!CompatLoadLibrary(&connID, name_dll2))
	{
		cvs_err("Error: Unable to load the library %s\n", name_dll2);
		return false;
	}

	// set the console out
	if(!CompatCallLibrary1(connID, result, uppSetGlueProcInfo, "dllglue_setconsoleout",
		dllglue_setglue_func, CompatFuncArg(uppConsoleOutProcInfo, consoleout)))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_setconsoleout""\n");
		goto closeandfail;
	}

	// set the console err
	if(!CompatCallLibrary1(connID, result, uppSetGlueProcInfo, "dllglue_setconsoleerr",
		dllglue_setglue_func, CompatFuncArg(uppConsoleOutProcInfo, consoleerr)))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_setconsoleerr""\n");
		goto closeandfail;
	}

	// set the console in
	if(!CompatCallLibrary1(connID, result, uppSetGlueProcInfo, "dllglue_setconsolein",
		dllglue_setglue_func, CompatFuncArg(uppConsoleInProcInfo, consolein)))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_setconsolein""\n");
		goto closeandfail;
	}

	// set the getenv
	if(!CompatCallLibrary1(connID, result, uppSetGlueProcInfo, "dllglue_setgetenv",
		dllglue_setglue_func, CompatFuncArg(uppGetEnvProcInfo, mygetenv)))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_setgetenv""\n");
		goto closeandfail;
	}

#ifdef macintosh
	// set the event dispatcher
	if(!CompatCallLibrary1(connID, result, uppSetGlueProcInfo, "dllglue_seteventdispatch",
		dllglue_setglue_func, CompatFuncArg(uppEventDispatchProcInfo, myeventdispatcher)))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_seteventdispatch""\n");
		goto closeandfail;
	}
#endif /* macintosh */

#ifdef qMacCvsPP
	CMacCvsApp::app->cvsLoaded = true;
#endif /* qMacCvsPP */
	
	return true;

closeandfail:
	CompatCloseLibrary(&connID);
	return false;
}

static void unloadCVS(CompatConnectID connID, int exitc)
{	
	CompatCloseLibrary(&connID);
	
	if(!gCvsPrefs.IsTclFileRunning())
		cvs_out("\n*****CVS exited normally with code %d*****\n\n", exitc);

#ifdef qMacCvsPP
	CMacCvsApp::app->cvsLoaded = false;
	CMacCvsApp::app->RefreshBrowsers();
#endif /* qMacCvsPP */
#ifdef WIN32
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	if(app->gCvsStopping)
	{
		cvs_out("*****CVS stopped on user request !!! *****\n\n");
	}
#endif /* WIN32 */
}
#endif /* !qUnix */

#ifdef WIN32
// critical section to protect while accessing MFC from the DLL
CRITICAL_SECTION CCvsThread::m_lock;

CCvsThread::CCvsThread()
{
}

void CCvsThread::Delete()
{
	// calling the base here won't do anything but it is a good habit
	CWinThread::Delete();
}

void CCvsThread::KillThread()
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();
	app->gCvsStopping = true;
}

CCvsThread::~CCvsThread()
{
	sLastWin32CvsThread = 0L;
}

BOOL CCvsThread::InitInstance()
{
	CWincvsApp* app = (CWincvsApp *)AfxGetApp();

	if(!loadCVS(mConnID))
		return FALSE;
	
	app->gCvsRunning = true;
	app->gCvsStopping = false;

	app->gLastCvsResult = Launch();

	unloadCVS(mConnID, app->gLastCvsResult);

	app->gCvsRunning = false;
	app->gCvsStopping = false;

	app->GetFileView()->CheckChanges();

	return FALSE;
}

IMPLEMENT_DYNAMIC(CCvsThread, CWinThread);

BEGIN_MESSAGE_MAP(CCvsThread, CWinThread)
	//{{AFX_MSG_MAP(CCvsThread)
		// NOTE - the ClassWizard will add and remove mapping macros here.
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

class CCvsCmdThread : public CCvsThread
{
public:
	DECLARE_DYNAMIC(CCvsCmdThread)

	CCvsCmdThread(const char *path, int argc, char * const *argv);
	virtual ~CCvsCmdThread();

// Overrides
	// ClassWizard generated virtual function overrides
	//{{AFX_VIRTUAL(CCvsCmdThread)
	//}}AFX_VIRTUAL

protected:
	virtual int Launch();

	CvsArgs mArgs;
	CPStr mPath;

	// Generated message map functions
	//{{AFX_MSG(CCvsCmdThread)
		// NOTE - the ClassWizard will add and remove member functions here.
	//}}AFX_MSG

	DECLARE_MESSAGE_MAP()
};

CCvsCmdThread::CCvsCmdThread(const char *path, int argc, char * const *argv) :
		mArgs(argv, argc), mPath(path), CCvsThread()
{
}

CCvsCmdThread::~CCvsCmdThread()
{
}

int CCvsCmdThread::Launch()
{
	int exitc = 0;

	if(!CompatCallLibrary3(mConnID, exitc, uppMainProcInfo, "dllglue_main",
		dllglue_main_func, mPath.empty() ? 0L : (const char *)mPath,
		mArgs.Argc(), (char **)mArgs.Argv()))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_main""\n");
		exitc = 1;
	}

	return exitc;
}

IMPLEMENT_DYNAMIC(CCvsCmdThread, CCvsThread);

BEGIN_MESSAGE_MAP(CCvsCmdThread, CCvsThread)
	//{{AFX_MSG_MAP(CCvsCmdThread)
		// NOTE - the ClassWizard will add and remove mapping macros here.
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

#endif /* WIN32 */

int launchCVS(const char *path, int argc, char * const *argv, CCvsConsole * console)
{
	sConsole = console;

#ifndef qUnix	
#ifdef WIN32
	if(console != 0L)
	{
	sLastWin32CvsThread = 0L;
#endif /* WIN32 */
	CompatConnectID connID;
	
	if(!loadCVS(connID))
		return errInternal;
	
	int exitc = 0;

	// call the main
	if(!CompatCallLibrary3(connID, exitc, uppMainProcInfo, "dllglue_main",
		dllglue_main_func, path, argc, (char **)argv))
	{
		cvs_err("Error : Cannot find symbol ""dllglue_main""\n");
		exitc = errInternal;
	}

	unloadCVS(connID, exitc);
	
	return exitc;
#ifdef WIN32
	}
	else
	{
	CCvsCmdThread* pThread;
	pThread = new CCvsCmdThread(path, argc, (char **)argv);
	if (pThread == 0L)
	{
		cvs_err("Impossible to create the cvs thread !\n");
		return errInternal;
	}

	ASSERT_VALID(pThread);

	// Create Thread in a suspended state so we can set the Priority 
	// before it starts getting away from us
	if (!pThread->CreateThread(CREATE_SUSPENDED))
	{
		cvs_err("Impossible to start the cvs thread !\n");
		delete pThread;
		return errInternal;
	}

	// If you want to make the sample more sprightly, set the thread priority here 
	// a little higher. It has been set at idle priority to keep from bogging down 
	// other apps that may also be running.
	VERIFY(pThread->SetThreadPriority(THREAD_PRIORITY_NORMAL));
	// Now the thread can run wild
	pThread->ResumeThread();
	sLastWin32CvsThread = pThread;

	return errThreadRunning;
	}
#endif /* WIN32 */
#else /* qUnix */
	static CStr cvscmd;

	if(UCvsApp::gApp->IsCvsRunning())
	{
		cvs_err("Error : cvs is already running !\n");
		return errInternal;
	}

	if(path != 0L && chdir(path) != 0)
	{
		cvs_err("Error : cannot to chdir to %s (error %d)!\n", path, errno);
	}

	UCvsApp::gApp->SetCvsStopping(false);

	struct stat sb;
	cvscmd = UCvsApp::gApp->GetAppLibFolder();
	if(!cvscmd.endsWith(kPathDelimiter))
		cvscmd << kPathDelimiter;
	cvscmd << "cvs";
	UStr firstCmd = cvscmd;
	if(stat(cvscmd, &sb) == -1 || !(S_ISREG(sb.st_mode) || S_ISLNK(sb.st_mode)))
	{
		cvscmd = PACKAGE_SOURCE_DIR;
		CStr uppath, folder;
		SplitPath(cvscmd, uppath, folder);
		cvscmd = uppath;
		if(!cvscmd.endsWith(kPathDelimiter))
			cvscmd << kPathDelimiter;
		cvscmd << "cvsunix/src/cvs";
	}

	if(stat(cvscmd, &sb) == -1 || !(S_ISREG(sb.st_mode) || S_ISLNK(sb.st_mode)))
	{
		cvs_err("Unable to find the cvs process in : (error %d)\n", errno);
		cvs_err("1) %s\n", (const char *)firstCmd);
		cvs_err("2) %s\n", (const char *)cvscmd);
	}

	CvsProcess *proc = cvs_process_open(cvscmd, argc - 1, (char **)argv + 1, &sCallTable);
	if(proc == 0L)
	{
		cvs_err("Unable to initialize the cvs process (error %d)\n", errno);
		cvs_err("The cvs used is : %s\n", (const char *)cvscmd);
	}
	else if(console != 0L)
	{
		while(1)
		{
			cvs_process_give_time();
			if(!cvs_process_is_active(proc))
				break;
		}
	}
	else
	{
		UCvsApp::gApp->SetCvsRunning(true);
		UCvsApp::gCurCvs = proc;
	}

	return 0;
#endif /* qUnix */
}

class CStreamConsole : public CCvsConsole
{
public:
	CStreamConsole(FILE *out) : fOut(out) {}

	virtual long cvs_out(char *txt, long len)
	{
		fwrite(txt, sizeof(char), len, fOut);
		return len;
	}
	virtual long cvs_err(char *txt, long len)
	{
		cvs_errstr(txt, len);
		return len;
	}
protected:
	FILE *fOut;
};

FILE *launchCVS(const char *dir, const CvsArgs & args, CStr & tmpFile, const char *prefix, const char *extension)
{
	if(!MakeTmpFile(tmpFile, prefix, extension))
		return 0L;

	FILE * res = fopen(tmpFile, "w+");
	if(res == 0L)
	{
		cvs_err("Impossible to open for write the temp file '%s' (error %d)",
			(const char *)tmpFile, errno);
		return 0L;
	}

	CStreamConsole myconsole(res);

	launchCVS(dir, args.Argc(), args.Argv(), &myconsole);

	fseek(res, 0L, SEEK_SET);

	return res;
}

class CTmpFiles
{
public:
	~CTmpFiles()
	{
		std::vector<CStr>::const_iterator i;
		for(i = allfiles.begin(); i != allfiles.end(); ++i)
		{
			const char *file = *i;
			unlink((char *)file);
		}
	}

	inline const char *push_back(const char *file)
	{
		allfiles.push_back(file);
		return allfiles[allfiles.size() - 1];
	}
protected:
	std::vector<CStr> allfiles;
} sTmpList;

const char *launchCVS(const char *dir, const CvsArgs & args, const char *prefix, const char *extension)
{
	CStr filename;
	FILE *file = launchCVS(dir, args, filename, prefix, extension);
	if(file == 0L)
		return 0L;

	fclose(file);
	return sTmpList.push_back(filename);
}

#ifdef WIN32
void WaitForCvs(void)
{
	// pump messages manually
	while(1)
	{
		if(sLastWin32CvsThread == 0L)
			return;

		DWORD dw = ::WaitForSingleObject(*sLastWin32CvsThread, 300);

		switch(dw) 
		{ 
			case WAIT_OBJECT_0: 
			case WAIT_TIMEOUT: 
				break;
			case WAIT_ABANDONED: 
			default: 
				cvs_err("An error occured while waiting for cvs to return (error %d)\n", ::GetLastError());
				return;
				break;
		}

		// pump message, but quit on WM_QUIT
		if (!AfxGetThread()->PumpMessage())
		{
			AfxPostQuitMessage(0);
			return;
		}
	}
}
#endif

#ifdef macintosh
	long GUSIMSL_WriteConsoleHook(char *buffer, long n)
	{
		return consoleout(buffer, n);
	}
	long GUSIMSL_WriteErrHook(char *buffer, long n)
	{
		return consoleerr(buffer, n);
	}
	int GUSIMSL_HandleEventHook(EventRecord *ev)
	{
		return myeventdispatcher(ev);
	}
#endif /* macintosh */
